/*******************************************************************************
 * Copyright (c) 2016 - 2022 Intel Corporation and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Intel Corporation - Initial implementation, based on code taken and modified
 *     					   from org.eclipse.cdt.dsf.gdb.
 *******************************************************************************/
package org.eclipse.cdt.dsf.iss.service;

import java.util.Collections;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.cdt.dsf.concurrent.DataRequestMonitor;
import org.eclipse.cdt.dsf.concurrent.ImmediateRequestMonitor;
import org.eclipse.cdt.dsf.concurrent.RequestMonitor;
import org.eclipse.cdt.dsf.datamodel.AbstractDMContext;
import org.eclipse.cdt.dsf.datamodel.IDMContext;
import org.eclipse.cdt.dsf.gdb.service.extensions.GDBRunControl_HEAD;
import org.eclipse.cdt.dsf.iss.IssGdbPlugin;
import org.eclipse.cdt.dsf.iss.mi.service.command.IssCommandFactory;
import org.eclipse.cdt.dsf.iss.mi.service.command.output.MIFunctionRecord;
import org.eclipse.cdt.dsf.iss.mi.service.command.output.MIFunctionRecordInfoInfo;
import org.eclipse.cdt.dsf.iss.mi.service.command.output.MIFunctionRecordLengthInfo;
import org.eclipse.cdt.dsf.mi.service.IMICommandControl;
import org.eclipse.cdt.dsf.mi.service.IMIExecutionDMContext;
import org.eclipse.cdt.dsf.mi.service.command.events.MIBreakpointHitEvent;
import org.eclipse.cdt.dsf.mi.service.command.output.MIInfo;
import org.eclipse.cdt.dsf.service.DsfServiceEventHandler;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

/**
 * @since 4.2
 */
public class IssGDBRunControl extends GDBRunControl_HEAD implements IIssReverseRunControl {
	private IMICommandControl fCommandControl;
	private IssCommandFactory fCommandFactory;
	private boolean isHookEvent = false;

	public IssGDBRunControl(DsfSession session) {
		super(session);
	}

	@Override
	public void initialize(final RequestMonitor requestMonitor) {
		super.initialize(new ImmediateRequestMonitor(requestMonitor) {
			@Override
			public void handleSuccess() {
				doInitialize(requestMonitor);
			}
		});
	}

	private void doInitialize(final RequestMonitor requestMonitor) {

		fCommandControl = getServicesTracker().getService(IMICommandControl.class);
		fCommandFactory = (IssCommandFactory) fCommandControl.getCommandFactory();

		register(new String[] { IssGDBRunControl.class.getName(), IIssReverseRunControl.class.getName() },

				new Hashtable<String, String>());
		requestMonitor.done();

	}

	public class FunctionRecordDMData implements IFunctionRecordDMData {

		String FCHName;
		String FCHId;
		String FCHLevel;
		String FCHFile;
		int FCHLineBegin;
		int FCHLineEnd;

		FunctionRecordDMData(String name, String id, String level, String file, String lineBegin, String lineEnd) {
			FCHName = name;
			FCHId = id;
			FCHLevel = level;
			FCHFile = file;
			FCHLineBegin = Integer.parseInt(lineBegin);
			FCHLineEnd = Integer.parseInt(lineEnd);
		}

		@Override
		public String getName() {
			return FCHName;
		}

		@Override
		public String getId() {
			return FCHId;
		}

		@Override
		public String getLevel() {
			return FCHLevel;
		}

		@Override
		public String getFile() {
			return FCHFile;
		}

		@Override
		public int getBeginLine() {
			return FCHLineBegin;
		}

		@Override
		public int getEndLine() {
			return FCHLineEnd;
		}
	}

	public class FunctionRecordDMContext extends AbstractDMContext implements IFunctionRecordDMContext {

		String fchID;

		public FunctionRecordDMContext(DsfSession fSessionId, IDMContext[] parents, String id) {
			super(fSessionId, parents);
			fchID = id;
		}

		@Override
		public int getId() {
			return Integer.parseInt(fchID);
		}

		@Override
		public boolean equals(Object obj) {
			return baseEquals(obj) && ((FunctionRecordDMContext) obj).fchID.equals(fchID);
		}

		@Override
		public int hashCode() {
			return baseHashCode() ^ (fchID == null ? 0 : fchID.hashCode());
		}

	}

	private class ContainerFunctionRecord {
		private int begin = 0;
		private int end = 0;
		private int count = 0;
		private final int size = window;
		private List<IFunctionRecordDMData> data;
		private Comparator<IFunctionRecordDMData> comparator;

		public ContainerFunctionRecord() {
			data = new LinkedList<IFunctionRecordDMData>();
			comparator = new Comparator<IFunctionRecordDMData>() {
				@Override
				public int compare(IFunctionRecordDMData d1, IFunctionRecordDMData d2) {
					int t1 = Integer.parseInt(d1.getId());
					int t2 = Integer.parseInt(d2.getId());

					return (t1 - t2);
				}
			};
		}

		public int getBegin() {
			return begin;
		}

		public int getEnd() {
			return end;
		}

		public int getCount() {
			return count;
		}

		public IFunctionRecordDMData getDatafromContextId(int i) {

			IFunctionRecordDMData temp = new FunctionRecordDMData(null, String.valueOf(i), null, null, "0", "0"); //$NON-NLS-1$ //$NON-NLS-2$
			int retVal = Collections.binarySearch(data, temp, comparator);
			if (retVal >= 0)
				return data.get(retVal);

			return null;
		}

		public IFunctionRecordDMData getDatafromIndex(int i) {

			if (i < count)
				return data.get(i);

			return null;
		}

		public void clearData() {
			begin = 0;
			end = 0;
			count = 0;
			data.clear();
		}

		public void addAtBegin(IFunctionRecordDMData records[]) {
			int itemsToRemove = records.length + count - size;
			if (itemsToRemove > 0) {
				for (int i = count - itemsToRemove; i <= count - 1; i++)
					data.remove(data.size() - 1);
				for (int i = 0; (i < records.length) && (i < itemsToRemove); i++) {
					data.add(i, records[i]);
				}
			} else {
				for (int i = 0; (i < records.length); i++) {
					data.add(i, records[i]);
					count++;
				}
			}

			end = Integer.parseInt(data.get(count - 1).getId());
			begin = Integer.parseInt(data.get(0).getId());
		}

		public void addAtEnd(IFunctionRecordDMData records[]) {
			int itemsToRemove = records.length + count - size;
			if (itemsToRemove > 0) {
				for (int i = 0; i <= itemsToRemove - 1; i++)
					data.remove(0);
				for (int i = 0; (i < records.length) && (i < itemsToRemove); i++) {
					data.add(count - itemsToRemove + i, records[i]);
				}
			} else {
				int lastCount = count;
				for (int i = 0; (i < records.length); i++) {
					data.add(lastCount + i, records[i]);
					count++;
				}
			}

			end = Integer.parseInt(data.get(count - 1).getId());
			begin = Integer.parseInt(data.get(0).getId());

		}

		public void fillData(IFunctionRecordDMData records[]) {
			if (records.length > 0) {
				begin = Integer.parseInt(records[0].getId());
				for (int i = 0; (i < records.length) && (i < size); i++) {
					data.add(i, records[i]);
					end = Integer.parseInt(records[i].getId());
					count++;
				}
			}
		}
	}

	private ContainerFunctionRecord fchData = new ContainerFunctionRecord();

	private int fchWindowBegin = 0;

	private int fchWindowEnd = 0;

	private final int window = 100;

	private int fchRecordLength;

	private int minStackDepth;

	@Override
	public void getFunctionCallCountVM(IMIExecutionDMContext execDmc, final DataRequestMonitor<Integer> rm) {

		if (execDmc == null) {
			rm.done();
			return;
		}

		getConnection().queueCommand(fCommandFactory.createMIFunctionRecordLength(execDmc),
				new DataRequestMonitor<MIFunctionRecordLengthInfo>(getExecutor(), rm) {
					@Override
					protected void handleSuccess() {
						int length = getData().getFuncRecordLength();
						fchRecordLength = length;
						if (length > window) {
							rm.setData(window);
							fchWindowBegin = length - window + 1;
							fchWindowEnd = length;
						}

				else {
							rm.setData(length);
							fchWindowBegin = 1;
							fchWindowEnd = length;
						}
						fchData.clearData();
						rm.done();
					}

					@Override
					protected void handleFailure() {
						rm.setStatus(new Status(IStatus.ERROR, IssGdbPlugin.PLUGIN_ID, INVALID_HANDLE,
								"Could not get Function call history", null)); //$NON-NLS-1$
						rm.done();
					}
				});

	}

	@Override
	public void shiftWindowForward(int i) {
		if (fchData.getCount() > 0) {
			if (fchWindowEnd == fchRecordLength)
				return;

			if ((fchRecordLength - fchWindowEnd) >= i) {
				fchWindowEnd += i;
				fchWindowBegin += i;
			} else {
				fchWindowBegin = fchRecordLength - window + 1;
				fchWindowEnd = fchRecordLength;
			}
		}
	}

	@Override
	public void shiftWindowBackward(int i) {
		if (fchData.getCount() > 0) {
			if (fchWindowBegin == 1)
				return;

			if (fchWindowBegin > i) {
				fchWindowEnd -= i;
				fchWindowBegin -= i;
			} else {
				fchWindowBegin = 1;
				fchWindowEnd = (fchRecordLength > window ? window : fchRecordLength);
			}
		}
	}

	private IFunctionRecordDMData[] getDataFromMIOutput(MIFunctionRecord[] FCHData) {
		IFunctionRecordDMData records[] = new FunctionRecordDMData[FCHData.length];

		for (int i = 0; i < FCHData.length; i++) {

			String id = FCHData[i].getFuncIndex();
			String name = FCHData[i].getFuncName();
			String level = FCHData[i].getFuncLevel();
			String file = FCHData[i].getSrcloc();
			String lineBegin = FCHData[i].getSrcbegin();
			String lineEnd = FCHData[i].getSrcend();
			records[i] = new FunctionRecordDMData(name, id, level, file, lineBegin, lineEnd);
		}
		return records;
	}

	private void setMinStackDepth() {
		int min = Integer.MAX_VALUE;
		for (int i = 0; i < fchData.getCount(); i++) {
			IFunctionRecordDMData datafromid = fchData.getDatafromIndex(i);
			if (datafromid != null) {
				int temp = Integer.parseInt(datafromid.getLevel());
				min = (temp < min) ? temp : min;
			}
		}
		minStackDepth = min;
	}

	private IFunctionRecordDMContext[] getIFunctionRecordDMContextfromContainer(final IMIExecutionDMContext execDmc) {
		IFunctionRecordDMContext FCHData2[] = null;
		if (fchData.getCount() > 0) {
			FCHData2 = new FunctionRecordDMContext[fchData.getCount()];
			for (int i = 0; i < fchData.getCount(); i++) {
				IFunctionRecordDMData datafromid = fchData.getDatafromIndex(i);
				if (datafromid != null) {
					FCHData2[i] = new FunctionRecordDMContext(getSession(), new IDMContext[] { execDmc },
							datafromid.getId());
				}
			}
		}
		return FCHData2;
	}

	@Override
	public void getPartialFunctionCallRecordVM(final IMIExecutionDMContext execDmc,
			final DataRequestMonitor<IFunctionRecordDMContext[]> rm) {

		if (execDmc == null) {
			rm.done();
			return;
		}

		if (fchWindowBegin == fchData.getBegin() && fchWindowEnd == fchData.getEnd()) {
			rm.setData(getIFunctionRecordDMContextfromContainer(execDmc));
			rm.done();
			return;
		}

		boolean addAtBegintemp = false;
		final boolean overlap;
		int start = 0, end = 0;

		if (fchWindowBegin < fchData.getBegin() && fchWindowEnd > fchData.getBegin()) {
			addAtBegintemp = true;
			overlap = true;
			start = fchWindowBegin;
			end = fchData.getBegin() - 1;
		} else if (fchWindowEnd > fchData.getEnd() && fchWindowBegin < fchData.getEnd()) {
			addAtBegintemp = false;
			overlap = true;
			start = fchData.getEnd() + 1;
			end = fchWindowEnd;
		} else {
			overlap = false;
			start = fchWindowBegin;
			end = fchWindowEnd;
		}
		final boolean addAtBegin = addAtBegintemp; // In case of false, we add at end

		getConnection().queueCommand(fCommandFactory.createMIFunctionRecordInfo(execDmc, start, end),
				new DataRequestMonitor<MIFunctionRecordInfoInfo>(getExecutor(), rm) {
					@Override
					protected void handleSuccess() {
						IFunctionRecordDMData records[] = getDataFromMIOutput(getData().getFuncRecordList());

						if (overlap) {
							if (addAtBegin == true)
								fchData.addAtBegin(records);
							else
								fchData.addAtEnd(records);
						} else {
							fchData.clearData();
							fchData.fillData(records);
						}
						rm.setData(getIFunctionRecordDMContextfromContainer(execDmc));
						setMinStackDepth();
						rm.done();
					}

					@Override
					protected void handleFailure() {
						rm.setStatus(new Status(IStatus.ERROR, IssGdbPlugin.PLUGIN_ID, INVALID_HANDLE,
								"Could not get Function call history", null)); //$NON-NLS-1$
						rm.done();
					}
				});
	}

	@Override
	public void getFunctionCallRecordList(final DataRequestMonitor<IFunctionRecordDMData> rm, final int index) {

		if (fchData.getCount() > 0) {
			rm.setData(fchData.getDatafromContextId(index));
		} else {
			rm.setStatus(new Status(IStatus.ERROR, IssGdbPlugin.PLUGIN_ID, INVALID_HANDLE,
					"Could not get Function call history", null)); //$NON-NLS-1$
		}
		rm.done();
	}

	@Override
	public int getFunctionCallRecordMinStackDepth() {
		return minStackDepth;
	}

	/**
	 * OneAPI hack to overcome failure to continue after hook breakpoint is hit.
	 * We wait for the hook and flag it when it is hit. Only after the next thread,
	 * which is spawned by gdb-oneapi is suspended, we call continue. If we call continue
	 * right after hook bp is hit, it will fail because the second thread is not yet
	 * suspended.
	 * 
	 */
	@Override
	@DsfServiceEventHandler
	public void eventDispatched(ContainerSuspendedEvent e) {
		super.eventDispatched(e);

		var event = e.getMIEvent();
		if (event instanceof MIBreakpointHitEvent) {
			var bpEvent = (MIBreakpointHitEvent) event;
			var frame = bpEvent.getFrame();
			var func = frame.getFunction();
			if (func.equals("igfxdbgxchgDebuggerHook")) {
				isHookEvent = true;
			}
		} else {
			if (isHookEvent) {
				resume(e.getDMContext(), new DataRequestMonitor<MIInfo>(getExecutor(), null));
				isHookEvent = false;
			}
		}
	}

}
