/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::matmat to perform general
*       sparse matrix-sparse matrix multiplication on a SYCL device (Host, CPU, GPU).
*
*           C = op(A) * op(B)
*
*       where op() is defined by one of
*
*           oneapi::mkl::transpose::{nontrans,trans}
*
*       It uses the simplified API usage model where the user only handles memory
*       allocation for the final C Matrix, and any temporary workspaces are
*       handled by the library.

*       The supported floating point data types for matmat matrix data are:
*           float
*           double
*
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <CL/sycl.hpp>

// local includes
#include "../common/common_for_examples.hpp"
#include "common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Sparse Matrix Multiply consisting of
// initialization of A and B matrices through process of creating C matrix as
// the product
//
// C = op(A) * op(B)
//
// In this case, we multiply a square symmetric A by itself to have C be the
// square of A.
//
template <typename fpType, typename intType>
int run_sparse_matrix_sparse_matrix_multiply_example(const cl::sycl::device &dev)
{
    // Initialize data for Sparse Matrix - Sparse Matrix Multiply
    oneapi::mkl::transpose opA = oneapi::mkl::transpose::nontrans;
    oneapi::mkl::transpose opB = oneapi::mkl::transpose::nontrans;

    oneapi::mkl::sparse::matrix_view_descr viewA = oneapi::mkl::sparse::matrix_view_descr::general;
    oneapi::mkl::sparse::matrix_view_descr viewB = oneapi::mkl::sparse::matrix_view_descr::general;
    oneapi::mkl::sparse::matrix_view_descr viewC = oneapi::mkl::sparse::matrix_view_descr::general;

    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::one;
    oneapi::mkl::index_base c_index = oneapi::mkl::index_base::zero;

    // Catch asynchronous exceptions
    auto exception_handler = [](cl::sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (cl::sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    cl::sycl::queue main_queue(dev, exception_handler);
    cl::sycl::context cxt = main_queue.get_context();

    //
    // set up dimensions of matrix products
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = size * size * size;
    intType b_ncols = b_nrows;
    intType b_nnz   = 27 * b_nrows;
    intType c_nrows = size * size * size;
    intType c_ncols = c_nrows;
    // c_nnz is unknown at this point

    //
    // array memory management tools
    //
    std::vector<intType *> int_ptr_vec;
    std::vector<fpType *> fp_ptr_vec;
    std::vector<std::int64_t *> i64_ptr_vec;
    std::vector<void *> void_ptr_vec;

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fpType, mkl_allocator<fpType, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;

    // USM arrays for A matrix
    intType *a_rowptr, *a_colind;
    fpType *a_val;

    a_rowptr = (intType *)malloc_shared((a_nrows + 1) * sizeof(intType), dev, cxt);
    a_colind = (intType *)malloc_shared((a_nnz) * sizeof(intType), dev, cxt);
    a_val    = (fpType *)malloc_shared((a_nnz) * sizeof(fpType), dev, cxt);

    if (!a_rowptr || !a_colind || !a_val)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(a_rowptr);
    int_ptr_vec.push_back(a_colind);
    fp_ptr_vec.push_back(a_val);

    // copy data to USM arrays
    for (intType i = 0; i < a_nrows + 1; ++i) {
        a_rowptr[i] = ia[i];
    }
    for (intType i = 0; i < a_nnz; ++i) {
        a_colind[i] = ja[i];
        a_val[i]    = a[i];
    }

    //
    // setup B data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ib;
    std::vector<intType, mkl_allocator<intType, 64>> jb;
    std::vector<fpType, mkl_allocator<fpType, 64>> b;

    ib.resize(b_nrows + 1);
    jb.resize(27 * b_nrows);
    b.resize(27 * b_nrows);

    intType b_ind = b_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, ib, jb, b, b_ind);
    b_nnz = ib[b_nrows] - b_ind;

    // USM arrays for B matrix
    intType *b_rowptr, *b_colind;
    fpType *b_val;

    b_rowptr = (intType *)malloc_shared((b_nrows + 1) * sizeof(intType), dev, cxt);
    b_colind = (intType *)malloc_shared((b_nnz) * sizeof(intType), dev, cxt);
    b_val    = (fpType *)malloc_shared((b_nnz) * sizeof(fpType), dev, cxt);

    if (!b_rowptr || !b_colind || !b_val)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(b_rowptr);
    int_ptr_vec.push_back(b_colind);
    fp_ptr_vec.push_back(b_val);

    // copy data to USM arrays
    for (intType i = 0; i < b_nrows + 1; ++i) {
        b_rowptr[i] = ib[i];
    }
    for (intType i = 0; i < b_nnz; ++i) {
        b_colind[i] = jb[i];
        b_val[i]    = b[i];
    }

    //
    // setup C data locally in CSR format
    // note: we don't know c_nnz so can only prepare ic at this point
    //
    intType *c_rowptr = nullptr, *c_colind = nullptr;
    fpType *c_val = nullptr;

    c_rowptr = (intType *)malloc_shared((c_nrows + 1) * sizeof(intType), dev, cxt);

    if (!c_rowptr)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(c_rowptr);

    //
    // other workspaces and arrays for matmat process
    //
    std::int64_t *sizeTempBuffer = nullptr, *sizeTempBuffer2 = nullptr, *c_nnz = nullptr;
    void *tempBuffer = nullptr, *tempBuffer2 = nullptr;


    std::cout << "\n\t\tsparse::matmat parameters:\n";
    std::cout << "\t\t\topA = " << opA << std::endl;
    std::cout << "\t\t\topB = " << opB << std::endl;

    std::cout << "\t\t\tviewA = " << viewA << std::endl;
    std::cout << "\t\t\tviewB = " << viewB << std::endl;
    std::cout << "\t\t\tviewC = " << viewC << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tB_nrows = B_ncols = " << b_nrows << std::endl;
    std::cout << "\t\t\tC_nrows = C_ncols = " << c_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;
    std::cout << "\t\t\tC_index = " << c_index << std::endl;

    //
    // Execute Matrix Multiply
    //

    oneapi::mkl::sparse::matrix_handle_t A;
    oneapi::mkl::sparse::matrix_handle_t B;
    oneapi::mkl::sparse::matrix_handle_t C;

    oneapi::mkl::sparse::matmat_descr_t descr = NULL;
    oneapi::mkl::sparse::matmat_request req;

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);
        oneapi::mkl::sparse::init_matrix_handle(&C);

        oneapi::mkl::sparse::set_csr_data(A, a_nrows, a_ncols, a_index, a_rowptr, a_colind, a_val);
        oneapi::mkl::sparse::set_csr_data(B, b_nrows, b_ncols, b_index, b_rowptr, b_colind, b_val);

        //
        // only c_rowptr exists at this point in process so pass in nullptrs
        //
        oneapi::mkl::sparse::set_csr_data(C, c_nrows, c_ncols, c_index, c_rowptr, (intType *)nullptr, (fpType *)nullptr);

        //
        // initialize the matmat descriptor
        //
        oneapi::mkl::sparse::init_matmat_descr(&descr);
        oneapi::mkl::sparse::set_matmat_data(descr, viewA, opA, viewB, opB, viewC);

        //
        // Stage 1:  work estimation
        //
        // Note we are using the simplified form (passing nullptr's) here and in compute
        // where we let the library handle the allocation and lifetime of temporary
        // workspaces internally.
        //
        req = oneapi::mkl::sparse::matmat_request::work_estimation;
        auto ev1 =
                oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, nullptr, nullptr, {});

        //
        // Stage 2:  compute
        //
        req      = oneapi::mkl::sparse::matmat_request::compute;
        auto ev2 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, nullptr, nullptr,
                                               {ev1});

        //
        // Stage 3:  finalize
        //

        // Step 3.1  get nnz
        req                 = oneapi::mkl::sparse::matmat_request::get_nnz;
        std::int64_t *c_nnz = (std::int64_t *)(malloc_shared(1 * sizeof(std::int64_t), dev, cxt));
        if (!c_nnz)
            throw std::runtime_error("Failed to allocate USM memory");
        i64_ptr_vec.push_back(c_nnz);
        auto ev3_1 =
                oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, c_nnz, nullptr, {ev2});

        // Step 3.2  allocate final c matrix arrays
        ev3_1.wait();
        c_colind = (intType *)malloc_shared(c_nnz[0] * sizeof(intType), dev, cxt);
        c_val    = (fpType *)malloc_shared(c_nnz[0] * sizeof(fpType), dev, cxt);
        if (!c_colind || !c_val) {
            throw std::bad_alloc();
        }
        int_ptr_vec.push_back(c_colind);
        fp_ptr_vec.push_back(c_val);
        oneapi::mkl::sparse::set_csr_data(C, c_nrows, c_ncols, c_index, c_rowptr, c_colind, c_val);

        // Step 3.3  finalize into C matrix
        req        = oneapi::mkl::sparse::matmat_request::finalize;
        auto ev3_3 = oneapi::mkl::sparse::matmat(main_queue, A, B, C, req, descr, nullptr, nullptr,
                                                 {ev3_1});

        // print out first bit of C solution
        sycl::event ev4 = main_queue.submit([&](sycl::handler &cgh) {
            cgh.depends_on({ev3_3});
            auto kernel = [=]() {
                std::cout << "C matrix [first two rows]:" << std::endl;
                intType c_ind = c_index == oneapi::mkl::index_base::zero ? 0 : 1;
                for (intType row = 0; row < std::min(static_cast<intType>(2), c_nrows); ++row) {
                    for (intType j = c_rowptr[row] - c_ind; j < c_rowptr[row + 1] - c_ind; ++j) {
                        intType col = c_colind[j];
                        fpType val  = c_val[j];
                        std::cout << "C(" << row + c_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });

        main_queue.wait();

        // clean up
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        oneapi::mkl::sparse::release_matrix_handle(&A);
        oneapi::mkl::sparse::release_matrix_handle(&B);
        oneapi::mkl::sparse::release_matrix_handle(&C);
    }
    catch (cl::sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        oneapi::mkl::sparse::release_matrix_handle(&A);
        oneapi::mkl::sparse::release_matrix_handle(&B);
        oneapi::mkl::sparse::release_matrix_handle(&C);

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        oneapi::mkl::sparse::release_matmat_descr(&descr);
        oneapi::mkl::sparse::release_matrix_handle(&A);
        oneapi::mkl::sparse::release_matrix_handle(&B);
        oneapi::mkl::sparse::release_matrix_handle(&C);

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }

    cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Sparse Matrix Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    C = op(A) * op(B)" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR format, and C is the\n"
                 "# sparse matrix product in CSR format"
              << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::matmat" << std::endl;
    std::cout << "#   sparse::init_matmat_descr" << std::endl;
    std::cout << "#   sparse::set_matmat_data" << std::endl;
    std::cout << "#   sparse::release_matmat_descr" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_host -- only runs host implementation
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: host, cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        if (*it == my_sycl_device_types::host_device) {
            std:: cout << "Host device is not currently supported by sparse::matmat" << std::endl;
            break;
        }

        cl::sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_multiply_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<cl::sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_multiply_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    return status;
}
