/*******************************************************************************
* Copyright 2018-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            oneapi::mkl::vm::expint1 example program text (SYCL interface)
 *
 *******************************************************************************/

#include <CL/sycl.hpp>
#include <algorithm>
#include <cstdint>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <map>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl/types.hpp"
#include "oneapi/mkl/vm.hpp"

#include "common_for_examples.hpp"

#include "vml_common.hpp"

static constexpr int VLEN = 4;

static ulp_table_type ulp_table = {
    {MAX_HA_ULP_S, 4.5}, {MAX_LA_ULP_S, 5.0}, {MAX_EP_ULP_S, 5.0E3},
    {MAX_HA_ULP_D, 2.0}, {MAX_LA_ULP_D, 5.0}, {MAX_EP_ULP_D, 7.0E7},
    {MAX_HA_ULP_C, 2.0}, {MAX_LA_ULP_C, 5.0}, {MAX_EP_ULP_C, 5.0E3},
    {MAX_HA_ULP_Z, 2.0}, {MAX_LA_ULP_Z, 5.0}, {MAX_EP_ULP_Z, 7.0E7},
};

//!
//! @brief Accuracy test
//!
template <typename A, typename R>
bool vExpInt1AccuracyLiteTest(const sycl::device &dev) {

    int argtype = ARG1_RES1;
    // *************************************************************
    // Data table declaraion
    // *************************************************************
data_2_t data
{
    .i = 0
,

    .data_f32 = std::vector<data_2_f32_t>

    {

{ UINT32_C(0x4086F102), UINT32_C(0x3B3EA496) }, //  0: vsExpInt1 ( 4.2169199       ) = ( 0.00290898001   );
{ UINT32_C(0x40021418), UINT32_C(0x3D3F82AE) }, //  1: vsExpInt1 ( 2.03247643      ) = ( 0.0467554852    );
{ UINT32_C(0x407BFADC), UINT32_C(0x3B85A594) }, //  2: vsExpInt1 ( 3.93718624      ) = ( 0.00407857634   );
{ UINT32_C(0x40806539), UINT32_C(0x3B740114) }, //  3: vsExpInt1 ( 4.01235628      ) = ( 0.00372320879   );
 }

,
    .data_f64 = std::vector<data_2_f64_t>

    {

{ UINT64_C(0x4010DE203A4CEF74), UINT64_C(0x3F67D492E5CCFD17) }, //  0: vdExpInt1 ( 4.21691981405807681       ) = ( 0.00290898028325127121    );
{ UINT64_C(0x40004282F4C21EA0), UINT64_C(0x3FA7F055E5D8C495) }, //  1: vdExpInt1 ( 2.03247634141355604       ) = ( 0.0467554896425862801     );
{ UINT64_C(0x400F7F5B79FEFEB7), UINT64_C(0x3F70B4B29A274FD2) }, //  2: vdExpInt1 ( 3.93718619641716883       ) = ( 0.00407857672185581339    );
{ UINT64_C(0x40100CA71821B2E8), UINT64_C(0x3F6E8022BE9F9EE1) }, //  3: vdExpInt1 ( 4.01235616403275941       ) = ( 0.00372320924815371145    );
 }

,

    .data_c32 = std::vector <data_2_c32_t>

    { /* empty */ }

,
    .data_c64 = std::vector <data_2_c64_t>

    { /* empty */ }

};


    // *************************************************************
    // Variable declaraions
    // *************************************************************
    // Input arguments
    A arg1;
    std::vector<A> varg1;
    // Output results
    R ref1;
    std::vector<R> vref1;
    std::vector<R> vres1;

    std::vector<R> vrefi1;
    std::vector<R> vresi1;
    // Number of errors
    int errs = 0;

    // *************************************************************
    // Vector input data initialization
    // *************************************************************
    for (int i = 0; i < VLEN; ++i) {
        // Getting values from reference data table

        data.get_values(arg1, ref1);

        // Pushing values into vectors
        varg1.push_back(arg1);
        vres1.push_back(777);
        vref1.push_back(ref1);

        vresi1.push_back(777);
        // Fill only uneven results for strided case
        if (i & 1) {
            vrefi1.push_back(ref1);
        } else {
            vrefi1.push_back(777);
        }
    } // for (int i = 0; i < VLEN; ++i)

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            } catch (sycl::exception const &e) {
                std::cerr << "Caught asynchronous SYCL exception:\n"
                          << e.what() << std::endl;
            }
        } // for (std::exception_ptr const& e : exceptions)
    };

    // Create execution queue with asynchronous error handling
    sycl::queue main_queue(dev, exception_handler);

    // Get device name
    std::string dev_name =
        main_queue.get_device().get_info<sycl::info::device::name>();

    // *************************************************************
    // Loop by all 3 accuracy modes of VM: HA, LA, EP:
    // set computation mode, run VM and check results
    // *************************************************************
    for (int acc = 0; acc < ACCURACY_NUM; ++acc) {
        // Clear result vectors
        std::fill(vres1.begin(), vres1.end(), 777);

        std::fill(vresi1.begin(), vresi1.end(), 777);
        // Create sycl buffers
        sycl::buffer<A, 1> in1(varg1.begin(), varg1.end());

        sycl::buffer<R, 1> out1(vres1.begin(), vres1.end());

        sycl::buffer<R, 1> outi1(vresi1.begin(), vresi1.end());
        // Run VM function
        oneapi::mkl::vm::expint1(main_queue, varg1.size(), in1, out1,
                                 accuracy_mode[acc]);

        // Strided VM calls for each uneven element
        {
            oneapi::mkl::vm::expint1(
                main_queue, in1, oneapi::mkl::slice(1, varg1.size() / 2, 2),
                outi1, oneapi::mkl::slice(1, vres1.size() / 2, 2),
                accuracy_mode[acc]);
        }

        // Get results from sycl buffers
        auto host_vres1 = out1.template get_access<sycl::access::mode::read>();

        auto host_vresi1 =
            outi1.template get_access<sycl::access::mode::read>();

        for (int i = 0; i < vres1.size(); ++i) {
            vres1[i] = host_vres1[i];

            vresi1[i] = host_vresi1[i];
        }

        // Catch sycl exceptions
        try {
            main_queue.wait_and_throw();
        } catch (sycl::exception const &e) {
            std::cerr << "SYCL exception during Accuracy Test\n"
                      << e.what() << " code: " << e.code().value() << std::endl;
            return false;
        }

        // *************************************************************
        // Compute ulp between computed and expected (reference)
        // values and check
        // *************************************************************
        for (int i = 0; i < VLEN; ++i) {
            // Check simple indexing function
            errs +=
                check_result<A, R>("", "expint1", ",simple", i, argtype, acc,
                                   varg1[i], varg1[i], vres1[i], vres1[i],
                                   vref1[i], vref1[i], ulp_table, false, false);

            // Check strided indexing function
            errs += check_result<A, R>("", "expint1", ",strided", i, argtype,
                                       acc, varg1[i], varg1[i], vresi1[i],
                                       vresi1[i], vrefi1[i], vrefi1[i],
                                       ulp_table, false, false);

        } // for (int i = 0; i < VLEN; ++i)
    }     // for (int acc = 0; acc < ACCURACY_NUM; ++acc)

    std::cout << "\tResult: " << ((errs == 0) ? "PASS" : "FAIL") << std::endl;

    return (errs == 0);
} // template <typename A, typename R> bool vFuncAccuracyText (const device
  // &dev)
//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner() {
    std::cout << "" << std::endl;
    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << "# General VM "
              << "expint1"
              << " Function Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   vm::"
              << "expint1" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;

    std::cout << "#   float" << std::endl;

    std::cout << "#   double" << std::endl;

    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_host -- only runs SYCL HOST device
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU device
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU device
// -DSYCL_DEVICES_all (default) -- runs on all: HOST, CPU and GPU devices
//
//  For each device selected and each data type supported, the example
//  runs with all supported data types
//
int main(int argc, char **argv) {
    // Number of errors occured
    int errs = 0;

    // Print standard banner for VM examples
    print_example_banner();

    // List of available devices
    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    // Loop by all available devices
    for (auto dev_type : list_of_devices) {
        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, dev_type);

        // Run tests if the device is available
        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[dev_type]
                      << ".\n";

            std::cout << "\tRunning with single precision real data type:"
                      << std::endl;
            errs += vExpInt1AccuracyLiteTest<float, float>(my_dev);

            std::cout << "\tRunning with double precision real data type:"
                      << std::endl;
            errs += vExpInt1AccuracyLiteTest<double, double>(my_dev);
        } else {

            std::cout << "No " << sycl_device_names[dev_type]
                      << " devices found; skipping "
                      << sycl_device_names[dev_type] << " tests.\n";
        }
    }

    return 0;
}
