/*******************************************************************************
* Copyright 2018-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            Cospi example program text (OpenMP offload interface)
 *
 *******************************************************************************/

#define VLEN 4

#include "_vml_common.h"

max_ulp_table_t max_ulp_table = 
{ //  HA   LA   EP
    { 4.5, 5.0, 5.0E3, }, // float
    { 2.0, 5.0, 7.0E7, }, // double
    { 2.0, 5.0, 5.0E3, }, // float complex
    { 2.0, 5.0, 7.0E7, }, // double complex
};

// device number
int dnum;

// *************************************************************
// Data table declaraion
// *************************************************************
data_2_t data =
{

{

{ { 0x40D9B85C }, { 0xBF50DE70 } }, //  0: vsCospi ( 6.80375481      ) = ( -0.815894127    );
{ { 0xC007309A }, { 0x3F70389A } }, //  1: vsCospi ( -2.1123414      ) = ( 0.938363671     );
{ { 0x40B52EFA }, { 0x3EF97341 } }, //  2: vsCospi ( 5.66198444      ) = ( 0.487207443     );
{ { 0x40BF006A }, { 0x3F7EC572 } }, //  3: vsCospi ( 5.96880054      ) = ( 0.995200276     );
}

,

{

{ { 0x401B370B60E66E18 }, { 0xBFEA1BCC308CE42E } }, //  0: vdCospi ( 6.80375434309419092       ) = ( -0.815893263649859923     );
{ { 0xC000E6134801CC26 }, { 0x3FEE07131CA9A12D } }, //  1: vdCospi ( -2.11234146361813924      ) = ( 0.938363605481787766      );
{ { 0x4016A5DF421D4BBE }, { 0x3FDF2E6881FFD0F4 } }, //  2: vdCospi ( 5.66198447517211711       ) = ( 0.487207533790992686      );
{ { 0x4017E00D485FC01A }, { 0x3FEFD8AE55A9C5B9 } }, //  3: vdCospi ( 5.96880066952146571       ) = ( 0.995200316725621481      );
}

,
{ /* empty */ }

,

{ /* empty */ }

};

//!
//! @brief Single precision test
//!

int vCospiAccuracyLiteTest_float() {
  int errs = 0;
  float *varg1 = (float *)malloc(VLEN * sizeof(float));
  float *vres1 = (float *)malloc(VLEN * sizeof(float));
  float *vref1 = (float *)malloc(VLEN * sizeof(float));

  float *vresi1 = (float *)malloc(VLEN * sizeof(float));
  float *vrefi1 = (float *)malloc(VLEN * sizeof(float));
  {
    for (int i = 0; i < VLEN; ++i) {
      varg1[i] = data.data_f32[i].v1.f;
      vref1[i] = data.data_f32[i].v2.f;

      vresi1[i] = ((i & 1) == 0) ? (float)(999) : (float)(777);
      vrefi1[i] = ((i & 1) == 0) ? data.data_f32[i].v2.f : (float)(777);
    }
  }

  for (int acc = 0; acc < ACCURACY_NUM; ++acc) {
    // Simple indexing function call
#pragma omp target data map(to:varg1[0:VLEN]) map(tofrom:vres1[0:VLEN]) device(dnum)
    {
#pragma omp target variant dispatch device(dnum) use_device_ptr(varg1, vres1) nowait
        {vmsCospi(VLEN, (const float *)varg1, (float *)vres1,
                  accuracy_mode[acc]);
  }
#pragma omp taskwait
}

// Strided indexing function call
{
#pragma omp target data map(to:varg1[0:VLEN]) map(tofrom:vresi1[0:VLEN]) device(dnum)
  {
#pragma omp target variant dispatch device(dnum) use_device_ptr(varg1, vresi1) nowait
    {
      vmsCospiI(VLEN / 2, (const float *)varg1, 2, (float *)vresi1, 2,
                accuracy_mode[acc]);
    }
#pragma omp taskwait
  }
}

for (int i = 0; i < VLEN; ++i) {
  // Check simple indexing function
  errs +=
      check_result_float(i, ARG1_RES1, varg1[i], varg1[i], vres1[i], vres1[i],
                         vref1[i], vref1[i], "Cospi", ", simple", acc);

  // Check strided indexing function
  errs +=
      check_result_float(i, ARG1_RES1, varg1[i], varg1[i], vresi1[i], vresi1[i],
                         vrefi1[i], vrefi1[i], "Cospi", ",strided", acc);
}
}

free(varg1);

free(vres1);
free(vref1);

free(vresi1);
free(vrefi1);
return errs;
}
//!
//! @brief Double precision test
//!

int vCospiAccuracyLiteTest_double() {
  int errs = 0;
  double *varg1 = (double *)malloc(VLEN * sizeof(double));
  double *vres1 = (double *)malloc(VLEN * sizeof(double));
  double *vref1 = (double *)malloc(VLEN * sizeof(double));

  double *vresi1 = (double *)malloc(VLEN * sizeof(double));
  double *vrefi1 = (double *)malloc(VLEN * sizeof(double));
  {
    for (int i = 0; i < VLEN; ++i) {
      varg1[i] = data.data_f64[i].v1.f;
      vref1[i] = data.data_f64[i].v2.f;

      vresi1[i] = ((i & 1) == 0) ? (double)(999) : (double)(777);
      vrefi1[i] = ((i & 1) == 0) ? data.data_f64[i].v2.f : (double)(777);
    }
  }

  for (int acc = 0; acc < ACCURACY_NUM; ++acc) {
    // Simple indexing function call
#pragma omp target data map(to:varg1[0:VLEN]) map(tofrom:vres1[0:VLEN]) device(dnum)
    {
#pragma omp target variant dispatch device(dnum) use_device_ptr(varg1, vres1) nowait
        {vmdCospi(VLEN, (const double *)varg1, (double *)vres1,
                  accuracy_mode[acc]);
  }
#pragma omp taskwait
}

// Strided indexing function call
{
#pragma omp target data map(to:varg1[0:VLEN]) map(tofrom:vresi1[0:VLEN]) device(dnum)
  {
#pragma omp target variant dispatch device(dnum) use_device_ptr(varg1, vresi1) nowait
    {
      vmdCospiI(VLEN / 2, (const double *)varg1, 2, (double *)vresi1, 2,
                accuracy_mode[acc]);
    }
#pragma omp taskwait
  }
}

for (int i = 0; i < VLEN; ++i) {
  // Check simple indexing function
  errs +=
      check_result_double(i, ARG1_RES1, varg1[i], varg1[i], vres1[i], vres1[i],
                          vref1[i], vref1[i], "Cospi", ", simple", acc);

  // Check strided indexing function
  errs += check_result_double(i, ARG1_RES1, varg1[i], varg1[i], vresi1[i],
                              vresi1[i], vrefi1[i], vrefi1[i], "Cospi",
                              ",strided", acc);
}
}

free(varg1);

free(vres1);
free(vref1);

free(vresi1);
free(vrefi1);
return errs;
}

int main(int argc, char **argv) {
  int errs = 0;
  int total_errs = 0;

  printf("Running %s functions:\n", "Cospi");

  printf("\tRunning %s with single precision real data type:\n", "Cospi");
  errs = vCospiAccuracyLiteTest_float();
  printf("\t%s single precision real result: %s\n\n", "Cospi",
         (errs == 0) ? "PASS" : "FAIL");
  total_errs += errs;

  printf("\tRunning %s with double precision real data type:\n", "Cospi");
  errs = vCospiAccuracyLiteTest_double();
  printf("\t%s double precision real result: %s\n\n", "Cospi",
         (errs == 0) ? "PASS" : "FAIL");
  total_errs += errs;
  printf("%s function result: %s\n\n", "Cospi",
         (total_errs == 0) ? "PASS" : "FAIL");

  return total_errs;
}
