"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isValidTenant = isValidTenant;
exports.resolve = resolve;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL, 'Global'];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param username
 * @param roles
 * @param availabeTenants
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param multitenancyEnabled
 * @param privateTenantEnabled
 * @param defaultTenant
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant({
  request,
  username,
  roles,
  availabeTenants,
  config,
  cookie,
  multitenancyEnabled,
  privateTenantEnabled,
  defaultTenant
}) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _request$url3, _request$url3$searchP, _config$multitenancy, _config$multitenancy2;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant'); // eslint-disable-next-line @typescript-eslint/naming-convention

  const security_tenant = request === null || request === void 0 ? void 0 : (_request$url3 = request.url) === null || _request$url3 === void 0 ? void 0 : (_request$url3$searchP = _request$url3.searchParams) === null || _request$url3$searchP === void 0 ? void 0 : _request$url3$searchP.get('security_tenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else if (defaultTenant && multitenancyEnabled) {
    selectedTenant = defaultTenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, multitenancyEnabled, privateTenantEnabled);
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, multitenancyEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (!multitenancyEnabled) {
    if (!globalTenantEnabled) {
      return undefined;
    }

    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  }
  /**
   * Fall back to the first tenant in the available tenants
   * Under the condition of enabling multitenancy, if the user has disabled both 'Global' and 'Private' tenants:
   * it will remove the default global tenant key for custom tenant.
   */


  if (Object.keys(availableTenantsClone).length > 1 && availableTenantsClone.hasOwnProperty(_common.globalTenantName)) {
    delete availableTenantsClone[_common.globalTenantName];
  }

  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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