package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````0!X```````!``````````+BQ````
M`````````$``.``)`$``&@`9``8````$````0`````````!``````````$``
M````````^`$```````#X`0````````@``````````P````0````X`@``````
M`#@"````````.`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````%B=````````6)T`````````
M``$```````$````&````>*L```````!XJP$``````'BK`0``````$`4`````
M```X!0```````````0```````@````8```"(JP```````(BK`0``````B*L!
M```````@`@```````"`"````````"``````````$````!````%0"````````
M5`(```````!4`@```````$0`````````1``````````$`````````%#E=&0$
M````<)4```````!PE0```````'"5````````!`$````````$`0````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````$`````````!2Y71D!````'BK````````>*L!``````!XJP$`
M`````(@$````````B`0````````!`````````"]L:6(O;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ```$````%`````,```!'3E4`U(H_WGU3@R6(ZAO!%+N)CI_,
M^#0$````$`````$```!'3E4```````,````'`````````"X```!)````!```
M``@````A$!``!<`$A"@"$`6`AD"$@@D`!`D@!(B(@!!!N@`""$D`````````
M2P```$T`````````3@````````!/````4`````````!1````4@```%,```!4
M`````````%8``````````````%<`````````6```````````````````````
M``!9````6@```%L```!<`````````%T``````````````%\`````````8```
M`&$```````````````````!B````8P````````!D```````````````NQ9NV
M:W^:?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/
MWD<MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+
MX\"S`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````````
M`````````````P`+`$`;`````````````````````````P`6``"P`0``````
M```````````H`0``$@````````````````````````#>`0``$@``````````
M``````````````"`````$@````````````````````````"P````$@``````
M``````````````````!Y`@``$@````````````````````````#E`0``$@``
M``````````````````````!E`0``$@`````````````````````````!````
M(`````````````````````````"7`@``$@````````````````````````"A
M`0``$@````````````````````````"7`P``$@``````````````````````
M``"G````$@````````````````````````#:`@``$0``````````````````
M``````!3````$@````````````````````````!:`@``$@``````````````
M``````````"[````$@`````````````````````````M`@``$@``````````
M``````````````"2`@``$@````````````````````````!A`@``$@``````
M```````````````````U`@``$@````````````````````````"&`0``(@``
M```````````````````````0````(``````````````````````````2`@``
M$@`````````````````````````(`0``$@`````````````````````````L
M````(``````````````````````````)`@``$@``````````````````````
M``#S`0``$@````````````````````````!J`@``$@``````````````````
M``````"Q`@``$@````````````````````````!3`@``$0``````````````
M```````````_`@``$@````````````````````````!X`0``$@``````````
M``````````````#U````%@````````````````````````"7````$@``````
M``````````````````!O````$@`````````````````````````9`@``$@``
M``````````````````````!4`0``$@````````````````````````#8````
M$@````````````````````````#[`0``$@````````````````````````"!
M`@``$@````````````````````````#(`@``$@``````````````````````
M``"5`0``$@````````````````````````#"`0``$@``````````````````
M``````#)````$@````````````````````````"@`@``$@``````````````
M``````````#G````$@`````````````````````````^`0``$@``````````
M``````````````!C````$@````````````````````````!I`@``$@``````
M``````````````````#0`0``$@````````````````````````!Q`@``$@``
M```````````````````````F`@``$@`````````````````````````3`0``
M$@````````````````````````":`0``$@`````````````````````````R
M`0``$@````````````````````````"X`@``$@``````````````````````
M```"`@``$@````````````````````````!&````$0``````````````````
M```````M`0``$@````````````````````````",````$@``````````````
M``````````!%`0``$@````````````````````````"H`0``$@``````````
M``````````````"P`0``$@````````````````````````"^`@``$0``````
M``````````````````#7`0``$@````````````````````````"_`@``(0``
M```````````````````````?`@``$@````````````````````````!%`@``
M$@````````````````````````#``@``$0````````````````````````#L
M`0``$@`````````````````````````7`P``$@`-`+`I````````!```````
M``"]`0``$@`-`,`;````````;`(```````"]`P``(``6``"P`0``````````
M```````)!```$@`-`(!0````````2`````````#6`P``$@`-`-!0````````
M'`$```````!4!```$@`-`&`N````````S`0````````D!```$``6`(BP`0``
M``````````````!*!```$0`6`("P`0``````"`````````#]`P``$``7`(BP
M`0````````````````#Q`P``$``7`+"P`0````````````````#R`P``$``7
M`+"P`0````````````````"%`P``$@`-`+`S````````,``````````7!```
M$@`-`$`U````````!`<```````#A`P``$@`-`&`H````````-``````````U
M`P``$@`-`$0\````````;`(```````#!`P``$@`-`$`>````````-```````
M``!9`P``$@`-`)0H````````%`$```````#I`P``$``7`+"P`0``````````
M``````"@`P``$@`-`+`^````````$`$```````"[`P``$``6``"P`0``````
M``````````#(`P``$``7`(BP`0`````````````````]!```$@`-`+0I````
M````N`(````````2!```$``7`+"P`0````````````````"L`P``$0`/``!4
M````````!`````````!+`P``$@`-`#`S````````>`````````![`P``$@`-
M`/!1````````V`$````````H`P``$@`-`'`L````````/`````````!O`P``
M$@`-`+`L````````J`$````````K!```$@`-`.`S````````8`$`````````
M7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%])
M5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?
M9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T
M7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?
M<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L7V-O;G-T
M<G5C=`!P97)L7V1E<W1R=6-T`%!,7V-U<G)E;G1?8V]N=&5X=`!P97)L7V%L
M;&]C`%!E<FQ?8W)O86M?;F]C;VYT97AT`'!E<FQ?9G)E90!097)L7W-A9F5S
M>7-M86QL;V,`4&5R;%]S>7-?:6YI=#,`4&5R;%]N97=84U]F;&%G<P!097)L
M7V%T9F]R:U]U;FQO8VL`4&5R;%]S>7-?=&5R;0!?7V-X85]F:6YA;&EZ90!R
M96%D`'-T<F-H<@!G971P:60`;65M;6]V90!?7VQI8F-?<W1A<G1?;6%I;@!?
M7V9P<FEN=&9?8VAK`'-T<F-M<`!O<&5N-C0`=6YL:6YK`'-T<G1O:P!M96US
M970`<W1R;F-M<`!G971U:60`;65M8VAR`&-L;W-E9&ER`'-T<G-T<@!R;61I
M<@!S=')L96X`<W1R9'5P`')E86QL;V,`<F5A9&1I<C8T`&%B;W)T`%]?<W!R
M:6YT9E]C:&L`<W1D97)R`&UE;6-P>0!S=')R8VAR`&QS=&%T-C0`;W!E;F1I
M<@!L<V5E:S8T`%]?97)R;F]?;&]C871I;VX`97AI=`!R96%D;&EN:P!?7W-T
M86-K7V-H:U]F86EL`&%C8V5S<P!M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E
M<E]A=&9O<FL`7U]S=&%C:U]C:&M?9W5A<F0`;&EB<&5R;"YS;P!L:6)C+G-O
M+C8`;&0M;&EN=7@M86%R8V@V-"YS;RXQ`'!A<E]C=7)R96YT7V5X96,`<&%R
M7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?96YV7V-L96%N
M`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S:&%?9FEN86P`
M9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7TE/7W-T
M9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`<VAA7W5P9&%T
M90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?<W1A<G0`<VAA
M7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P
M871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN:71?96YV`$=,24)#
M7S(N,3<`1TQ)0D-?,BXS-`!'3$E"0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N
M,S8O8V]R95]P97)L+T-/4D4```````````$``@`!``$``@`"``$``0`"``(`
M`@`!``,``0`"``$``@`"``(``@`"``$``@`!``$``@`"``0``@`"``(``0`!
M``$``0`"``$``0`"``(``@`"``(``0`"``$``@`!``0``@`"``(``0`"``$`
M`@`"``$``@`!``$``@`%``(``@`"``(``@`"``(``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!````
M``````$``0`!`P``$````"````"7D98&```#`&$$`````````0`#`/<"```0
M`````````+21E@8```4`;`0``!````"SD98&```$`'<$```0````EY&6!@``
M`@!A!````````'BK`0```````P0```````!0'P```````("K`0```````P0`
M````````'P```````%BO`0```````P0```````#04P```````/"O`0``````
M`P0```````#`&P````````BP`0```````P0````````(L`$``````""P`0``
M`````P0```````"P5P```````"BP`0```````P0```````!P5````````#BP
M`0```````P0```````!@50```````$"P`0```````P0```````"05P``````
M`$BP`0```````P0```````"85P```````%"P`0```````P0```````"@5P``
M`````%BP`0```````P0```````"H5P```````&BP`0```````P0```````!X
M5P```````'"P`0```````P0```````"`5P```````("P`0```````P0`````
M``"X5P```````."M`0```````00```,``````````````.BM`0```````00`
M``0``````````````/"M`0```````00```4``````````````/BM`0``````
M`00```8```````````````"N`0```````00```<```````````````BN`0``
M`````00```@``````````````!"N`0```````00```D``````````````!BN
M`0```````00```H``````````````""N`0```````00```L`````````````
M`"BN`0```````00```P``````````````#"N`0```````00```T`````````
M`````#BN`0```````00```X``````````````$"N`0```````00```\`````
M`````````$BN`0```````00``!```````````````%"N`0```````00``!$`
M`````````````%BN`0```````00``!(``````````````&"N`0```````00`
M`!,``````````````&BN`0```````00``!0``````````````'"N`0``````
M`00``!4``````````````'BN`0```````00``!8``````````````("N`0``
M`````00``!<``````````````(BN`0```````00``!@``````````````)"N
M`0```````00``!D``````````````)BN`0```````00``!H`````````````
M`*"N`0```````00``!L``````````````*BN`0```````00``!P`````````
M`````+"N`0```````00``!T``````````````+BN`0```````00``!X`````
M`````````,"N`0```````00``!\``````````````,BN`0```````00``"``
M`````````````-"N`0```````00``"(``````````````-BN`0``````!@0`
M`",``````````````."N`0```````00``"0``````````````.BN`0``````
M`00``"4``````````````/"N`0```````00``"8``````````````/BN`0``
M`````00``"<```````````````"O`0```````00``"@```````````````BO
M`0```````00``"D``````````````!"O`0```````00``"H`````````````
M`!BO`0```````00``"L``````````````""O`0```````00``"P`````````
M`````"BO`0```````00``"T``````````````#"O`0```````00``"X`````
M`````````#BO`0```````00``"\``````````````$"O`0```````00``#``
M`````````````$BO`0```````00``#$``````````````%"O`0```````00`
M`#(``````````````&"O`0```````00``#,``````````````&BO`0``````
M`00``#0``````````````'"O`0```````00``#4``````````````'BO`0``
M`````00``#8``````````````("O`0```````00``#<``````````````(BO
M`0```````00``#@``````````````)"O`0```````00``#D`````````````
M`)BO`0```````00``#H``````````````*"O`0```````00``#L`````````
M`````*BO`0```````00``#P``````````````+"O`0```````00``#T`````
M`````````+BO`0```````00``#X``````````````,"O`0```````00``#\`
M`````````````,BO`0```````00``$```````````````-"O`0```````00`
M`$,``````````````-BO`0```````00``$4``````````````."O`0``````
M`00``$8``````````````.BO`0```````00``$<``````````````/BO`0``
M`````00``$@``````````````,"M`0```````@0```H``````````````,BM
M`0```````@0``"$``````````````-"M`0```````@0``$$`````````````
M`!\@`]7]>[^I_0,`D<H``)3]>\&HP`-?U@``````````\'N_J=```+`1WD;Y
M$.(VD2`"']8?(`/5'R`#U1\@`]70``"P$>)&^1`"-Y$@`A_6T```L!'F1OD0
M(C>1(`(?UM```+`1ZD;Y$$(WD2`"']8`````````````````````_7NZJ<,`
M`+!C($?Y_0,`D?-3`:GU6P*IXH<#J>+C`)'A`P&1X$\`N>`S`9%D`$#YY"\`
M^00`@-+#``"P8^!'^6``/];"``"P0@A'^<```+``=$?YX0,"JL,``+!CK$?Y
M8``_UL```+``U$?Y``!`.8`*`#3```"P`&Q'^4'0.]73``#0(FA@^/1/0+E!
M*%(Y@!(`$90.`!$A!``R02@2.<$``+`AR$?Y`'Q]DR``/];C`P"JYB-`^<``
M`-!B0@*1Y4]`N01`0/DA``"0QP!`^2&@'9$@``"0`,`=D6<$`*E#!`#Y9``!
MJ;\$`''M"@!4(0"`TA\@`]4@#``1Q'AA^"$$`)%DV"#XOP`!:VS__U2E#``1
M=4("D7_8)?C%``"PI?Q&^>(#%"J@#D#Y`0``L`0`@-(A@!^1H``_UO0#`"H`
M!@`T=4("D?93`9'!``"P(:!'^:`.0/D@`#_6(```D.$#%JH`P!.1C@``E&`%
M`+1S0@*1P0``L"'P1OE@#D#Y(``_UL```+``:$?Y```_UL```+``($?YXB]`
M^0$`0/E"``'K`0"`TL$%`%3U6T*IX`,4*O-30:G]>\:HP`-?UL```+``3$?Y
MTP``T```/]9B0@*10`P`^<`#`+3!``"P(8!'^2``/];```"P`&Q'^4'0.]4B
M:&#X7]PH.:+__Q>@#D#YP0``L"$<1_D@`#_6]`,`*LS__Q>@"D#Y8```M#0(
M`)33__\7X0,6JB```)``X!&16P``E*`*`/F@^?^T+`@`E,O__Q>%`(!2LO__
M%\$``+`A-$?Y(`"`4B``/];```"P`)Q'^0``/]8?(`/5'R`#U1\@`]4?(`/5
M'R`#U1\@`]4=`(#2'@"`TN4#`*KA`T#YXB,`D>8#`)'```"P`/A'^0,`@-($
M`(#23?__ETC__Y?```"P``Q'^4```+1`__\7P`-?UA\@`]4?(`/5P```T``@
M`I'!``#0(2`"D3\``.O```!4P0``L"%$1_EA``"T\`,!J@`"'];``U_6P```
MT``@`I'!``#0(2`"D2$``,LB_'_300R!BR'\09/!``"TP@``L$)01_EB``"T
M\`,"J@`"'];``U_6'R`#U?U[OJG]`P"1\PL`^=,``-!@(D(Y0`$`-<$``+`A
M0$?Y@0``M,```-``!$#Y(``_UMC__Y<@`(!28"(".?,+0/G]>\*HP`-?UA\@
M`]4?(`/5W/__%Q\@`]4?(`/5'R`#U?U[NZG]`P"1\U,!J2`'`+3W8P.I]P,`
MJL```+``]$?Y^2,`^1D`0/F9!@"T]5L"J?8#`:H5`(!2X`)`.1_T`'$$&$!Z
M`0,`5#,#0/GS`0"TV```L!A;1_FU?D"3]`,9JN(#%:KA`Q>JX`,3J@`#/]:`
M```U8&IU.!_T`'$@`@!4DXY`^//^_[7U6T*I]V-#J?DC0/G@`Q.J\U-!J?U[
MQ:C``U_6]0,7JJ`>0#@?]`!Q!!A`>J'__U2U`A=+X___%X`"&<NU!@"1<P(5
MBP#\0Y/``@"YX`,3JO-30:GU6T*I]V-#J?DC0/G]>\6HP`-?UA,`@-+H__\7
M]V-#J1,`@-+Y(T#YY/__%Q\@`]4?(`/5_7NUJ<(``)!"($?Y_0,`D?,+`/GA
MHP"1\P,`JD,`0/GC5P#Y`P"`TL(``)!"L$?Y0``_UB`"`#7A.T"Y`@"44B$,
M%!(_$$!Q)!!">J`!`%3!``"0(2!'^>-70/DB`$#Y8P`"ZP(`@-+!`0!4\PM`
M^?U[RZC``U_6``"`4O7__Q?"``"00F!'^>`#$ZI!`(!20``_UA\``''@%Y\:
M[?__%\```)``G$?Y```_UA\@`]7!``"0(2!'^?U[LZG]`P"1(@!`^>)G`/D"
M`(#2P```M/-3`:GS`P"J`0!`.6$!`#7S4T&IP```D``@1_GB9T#Y`0!`^4(`
M`>L!`(#280L`5/U[S:C``U_6P0``D"&X1_D@`#_6]`,`JD#^_[3U6P*IU@``
MD-8^1_G@`Q2JP`(_UD`'`+0!3$`Y%4P`D3^X`'&!``!4H09`.0'__S0?(`/5
M`$Q`.1^X`'%@!P!4]V,#J>`#$ZK8``"0&.]'^0`#/];W`P"JX`,5J@`#/];!
M``"0(:1'^>`"``L`"``1`'Q`DR``/];&``"0QO!'^>4#%:H"`("2]0,`JB$`
M@%+D`Q.J`P``\&,@$)'``#_6P@``D$)<1_G@`Q6JX2,!D4``/]8?!``QH```
M5.!;0+D`#!02'Q!`<<`#`%3!``"0(?1&^>`#%:H@`#_6P0``D"'81_G@`Q6J
M(``_UO=C0ZG@`Q2JP`(_U@#Y_[7!``"0(51'^>`#%*H@`#_6P0``D"%X1_G@
M`Q.J(``_UO-30:GU6T*IJ?__%Z`&0#D?N`!Q@?C_5*`*0#F@]O\TP?__%^`#
M%:J4__^7Y?__%\```)``G$?Y\U,!J?5;`JGW8P.I```_UO]#0-'_`P+Y_X,!
MT0$`@-+#``"08R!'^>1#0)']>P"I_0,`D4(`@%+S4P&I\P,`*ODC`/G4``"0
ME`)'^64`0/F%+`#Y!0"`TH`"/]89!`#1X`,9ZR$_0)(`/$"2($2`VCD#`.O$
M!@!4]5L"J18!@-(V`*#RU0``D+621_GW8P.I]T,!D=@``+#A`QFJ`@"`4N`#
M$RJ``C_6X@,6JN$#%ZK@`Q,JH`(_U@!\0),?'`#QR0,`5`$@`-'A`@&KHP``
M5!H``!0A!`#1/P`7Z^,"`%0C`$#Y`@M`^7\``NM!__]4]5M"J2$`%\OW8T.I
M(``9B\$``)`A($?YXT-`D60L0/DB`$#YA``"ZP(`@-+!`0!4_7M`J?-30:GY
M(T#Y_X,!D?]#0)'``U_6.4-`T3]#0+'!^O]4]5M"J?=C0ZD``("2[/__%\``
M`)``G$?Y]5L"J?=C`ZD``#_6'R`#U1\@`]4,"(+2_V,LR\,``)!C($?YXO^!
MTOU[`*G]`P"1\U,!J?,#`*KT`P&J]5L"J?7C`)'A`Q6J8`!`^>`?"/D``(#2
MPP``D&,01_D```#P`$`0D6``/]8?``!QS0``5`0$`%'B`P`JH,ID.$`#`#7D
M`@`UP```D`!L1_E#T#O5X@,4JN$#$ZI@:&#XPP``D&,L1_E@`#_6P```D``@
M1_GB'TCY`0!`^4(``>L!`(#2@0,`5/U[0*D,"(+2\U-!J?5;0JG_8RR+P`-?
MUN(#!"H?(`/5PP``D&/01_E6?$"3X@,6JN`#%:KA!8!28``_UB#\_[3```"0
M`&Q'^430.]7C`Q:JX@,5JN$#$ZJ`:&#XQ```D(281_F``#_6W___%\```)``
MG$?Y```_UA\@`]4?(`/5_7NYJ?T#`)'S4P&I]`,!JO,#`*KU6P*I]V,#J?EK
M!*D@`$`YP0``D"$@1_D?]`!QUP``D/?N1_DB`$#YXC<`^0(`@-*4%I2:X`,4
MJN`"/];V`P"JX9,!D>`#$ZIA_O^78`@`M/4#`*K@`C_6WP(`ZRD'`%3W9X"Y
MV```D!CW1_G:``"06J='^??R?=-@`D`Y'_0`<0080'K`#`!4Y`,3JH,<0#A_
M]`!Q9!A`>J'__U2$`!/+A'Q`DQ4#0/G`"@"1```$BT`#/]:@:C?XH`(`M`$#
M0/G@`Q.J(VAW^`$40#AA```Y/_0`<2080'K```!4`A1`.&(<`#A?]`!Q1!A`
M>H'__U2A!X!2``"`TF$4`#B":F`X8F@@.``$`)&B__\UP```D``@1_GB-T#Y
M`0!`^4(``>L!`(#280@`5/-30:GU6T*I]V-#J?EK1*G]>\>HP`-?U@$`@-*"
M:F$XHFHA."$$`)&B__\U[O__%]@``)`8]T?Y^RL`^1L#0/E@`T#YP`4`M.`#
M&ZH5`(!2'R`#U0&,0/CB`Q4JM08`$:'__[5`#``1`'Q]D\$``+`BD$"Y@@(`
M-=H``)!:IT?Y(@"`4B*0`+E``S_6^0,`JD`$`+3#``"08^1'^;=^?9/A`QNJ
MX@,7JF``/]89`P#YH`8`$?5G`+D_VR#X^RM`^:3__Q?"``"00C!'^>$#`*K@
M`QNJ0``_U@`#`/GY`P"JP`$`M-H``)!:IT?YMWY]D_#__Q<$`(#2H?__%P`"
M@-(5`(!2VO__%\```)``G$?Y^RL`^0``/];[*T#YLO__%Q\@`]7]>[ZI(P``
M\`$``/#]`P"1\U,!J530.]73``"0<VY'^<(``)!"<$?Y8Z`4D<0``)"$W$?Y
M(8`0D8!J<_B``#_6@&IS^`0``/#S4T&IA(`=D?U[PJCC`P2JT```D!!^1_D"
M``"P`0``\$(0,9$A`!&1!0"`4@`"']8?(`/5'R`#U<```)``;$?Y1-`[U<(`
M`)!"<$?Y(P``\-```)`0WD?Y8Z`4D8!H8/@!``#P(8`0D0`"']8,"H32_V,L
MR\```)``($?Y`0B"TOU[`*G]`P"1\U,!J?/C`)'U$P#Y]6,ABP$`0/GA)Q#Y
M`0"`TL```)``%$?Y```_UN0#`"K&``"0QO!'^2(`@M(A`(!2X`,3J@4``/`#
M``#PI6`1D6.`$9'``#_6PP``D&,01_G@`Q.JX0,5J@(`@M)@`#_6(`3X-\$`
M`)`AI$?Y\P,`J@`$`!$`?$"3(``_UO0#`*H@`P"TPP``D&,H1_ES?D"3X0,5
MJN(#$ZI@`#_6GVHS.,```)``($?YXB=0^0$`0/E"``'K`0"`T@$!`%3]>T"I
MX`,4JO-30:D,"H32]1-`^?]C+(O``U_6P```D`"<1_D``#_6%`"`TN[__Q<?
M(`/5'R`#U;G__Q<,(H+2_V,LR\(``)!"($?Y_7L`J?T#`)'S4P&I]`,`JO5;
M`JGU`P&JX1,"D4``0/G@APCY``"`T@```/``X!&16_W_E_,#`*K"``"00L1'
M^>`#%*KA!8!20``_UF`"`+3A`Q2J````\``@$I'8_O^7X`,4JL$``)`A($?Y
MXX=(^2(`0/EC``+K`@"`TJ$0`%3]>T"I#"*"TO-30:GU6T*I_V,LB\`#7];!
M``"0(;Q'^?=C`ZG@`Q6JUP``D/<&1_G[<P6I%@``\.$_`/D@`#_6P6(2D>`"
M/];\`P"JP`T`M,```)``7$?Y^6L$J?@C!)':``"06O-'^>`W`/G5``"0M>Y'
M^<```)``M$?YX#L`^1\@`]6``T`Y8`<`--,``+3B.T#YX0,<JN`#$ZI``#_6
M@`4`-.`#'*J@`C_6`@0`T?L#%:J"`P*+^0,`JI\#`NNC``!4"@``%%_T'SB?
M`P+K@```5$``0#D?O`!Q8/__5.`#'*J@`C_6^0,`JN`#%*I@`S_6```9BP`$
M`)$?_#_Q*`,`5`(`@M(A`(!2Y`,<JN8#%*K@`QBJ!0``\`,``/"E@!*18Z`2
MD4`#/];B-T#YX2,"D>`#&*I``#_6H```->";0+D`#!02'R!`<>`"`%3!8A*1
M``"`TN`"/];\`P"J(/G_M?=C0ZGY:T2I^W-%J9C__Q?3`P"T8`)`.1^X`''@
M``!4!```\#D`@-+;``"0>^]'^9S`$9'5__\78`9`.8#]_S3X__\7P@``D$)@
M1_G@`QBJ(0"`4D``/]:@_/\UX0,8J@```/``(!*17/[_E^$_0/G@`QBJ(``_
MUO=C0ZGY:T2I^W-%J7___Q<$``#P.0"`TML``)![[T?YG,`1D;O__Q?W8T.I
M^W-%J7'__Q?```"0`)Q'^?=C`ZGY:P2I^W,%J0``/]8?(`/5_7N^J>$%@%+"
M``"00CA'^?T#`)'S"P#Y\P,`JD``/]8?``#Q8`:`FO,+0/G]>\*HT```D!"^
M1_D``A_6'R`#U0P(@M+_8RS+P0``D"$@1_G]>P"I_0,`D2(`0/GB'PCY`@"`
MT@`'`+3U$P#Y]0,`J@$`0#EA!@`TP0``D"'L1_GS4P&I(``_U@($`)'S`P"J
M7P1`\:@'`%3#``"08RA'^?3C`)'A`Q6JX`,4JF``/]9@!@#1@`(`BQ\`%.NH
M``!4!P``%!_T'S@?`!3K@```5`$`0#D_O`!Q8/__5,(``)!".$?YX`,4JN$%
M@%)``#_6``4`M!\`%.M(!`!4P0``D"&\1_G@`Q2J(``_UL$``)`A($?YXQ](
M^2(`0/EC``+K`@"`TB$%`%3]>T"I#`B"TO-30:GU$T#Y_V,LB\`#7];U$T#Y
MP```D``@1_GB'TCY`0!`^4(``>L!`(#2P0,`5-```)`0OD?Y````\`#`$9']
M>T"I#`B"TO]C+(L``A_6'P``.=[__Q<``(#2X/__%\```)``($?YXA](^0$`
M0/E"``'K`0"`T@$!`%3S4T&I````\/430/D`P!&1T```D!"^1_GK__\7P```
MD`"<1_D``#_6P```D`"<1_GS4P&I]1,`^0``/]8?(`/5'R`#U?U[NJG```"0
M`"!'^?T#`)'S4P&I%```\/-3`9'U6P*IE,(2D?=C`ZGY(P#YU0``D+7V1_D!
M`$#YX2\`^0$`@-+A`Q.JX`,4JB[\_Y?@`0"TH@)`^>-7@+E`#`.+`01`^4%X
M(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J(/S_EV#^_[75``"0M?9'
M^10``/"4`A.1X0,3JN`#%*H8_/^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!>"/X
MX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4J@K\_Y=@_O^UU0``D+7V1_D4
M``#PE"(3D>$#$ZK@`Q2J`OS_E^`!`+2B`D#YXU>`N4`,`XL!!$#Y07@C^.'^
M_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*KT^_^78/[_M=4``)"U]D?Y%```
M\)1B$Y'A`Q.JX`,4JNS[_Y?@`0"TH@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T
M`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2JWOO_EV#^_[76``"0UO9'^10``/"5
M@A.1X0,3JN`#%:K6^_^7X`$`M,("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(
M0/D!!`#Y`"``D:'__[7A`Q.JX`,5JLC[_Y=@_O^UM@``\-;V1_D5``#0M<(3
MD>$#$ZK@`Q6JP/O_E^`!`+3"`D#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y
M`00`^0`@`)&A__^UX0,3JN`#%:JR^_^78/[_M;<``/#W]D?Y%0``T+;B$9'A
M`Q.JX`,6JJK[_Y?@`0"TX@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$
M`/D`(`"1H?__M>$#$ZK@`Q:JG/O_EV#^_[6X``#P&/='^18``-#7`A21X0,3
MJN`#%ZJ4^_^7X`$`M`(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y
M`"``D:'__[7A`Q.JX`,7JH;[_Y=@_O^UN0``\#GW1_D7``#0^$(4D>$#$ZK@
M`QBJ?OO_E^`!`+0B`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@
M`)&A__^UX0,3JN`#&*IP^_^78/[_M;D``/`Y]T?Y&```T!B#%)'A`Q.JX`,8
MJFC[_Y?@`0"T(@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1
MH?__M>$#$ZK@`QBJ6OO_EV#^_[6Y``#P.?='^1@``-`8(Q*1X0,3JN`#&*I2
M^_^7X`$`M"(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'_
M_[7A`Q.JX`,8JD3[_Y=@_O^UX0,3J@```-``P!21/_O_E^$#`*I@``"TX$(4
MD</\_Y?A`Q.J````T``@%9$W^_^7X0,`JH```+0```#0`(`5D;K\_Y?A`Q.J
M````T`#`%9$N^_^7X0,`JF`"`+2@XA&1LOS_EZ```/``($?YXB]`^0$`0/E"
M``'K`0"`TF$"`%3U6T*I@((3D?-30:D!``#0]V-#J2%@%I'Y(T#Y_7O&J*+\
M_Q?A`Q.J````T```%I$6^_^7X0,`JF#]_[3``A21FOS_E^C__Q>@``#P`)Q'
M^0``/]8?(`/5H@``\$(@1_G]>[ZI````T/T#`)%#`$#YXP\`^0,`@-+A4P"1
M```4D0+[_Y?``0"T``!`.1_``'$$&$!ZX`>?&J$``/`A($?YXP]`^2(`0/EC
M``+K`@"`TJ$``%3]>\*HP`-?U@``@%+V__\7H```\`"<1_D``#_6'R`#U1\@
M`]6@``#P`(1'^?U[OZG]`P"1```_UJ$``/`A&$?Y(``_UD```+0``$#Y_7O!
MJ,`#7];]>[NIH0``\"$@1_G]`P"1\U,!J?0#`*KU6P*I%0``T+:"%I$B`$#Y
MXB<`^0(`@-+@`Q:JX1,!D=+Z_Y>```"T`0!`.?,#`*K!`0`UH```\``@1_GB
M)T#Y`0!`^4(``>L!`(#280<`5.$#%*J@@A:1\U-!J?5;0JG]>\6H2?S_%Z(`
M`/!"2$?YX0,4JD``/]:``0"TH```\``@1_GB)T#Y`0!`^4(``>L!`(#2(04`
M5/-30:GU6T*I_7O%J,`#7];W&P#YX`,4JK<``/#W[D?YX`(_UO4#`*K@`Q.J
MX`(_UJ$``/`AI$?YH`(`BP`(`)$@`#_6IP``\.?P1_GF`Q.J`@"`DB$`@%+D
M`Q2J\P,`J@4``-`#``#0I6`2D6.@$I'@`#_6H```\``@1_GB)T#Y`0!`^4(`
M`>L!`(#2P0``5/<;0/GA`Q.JX`,6JLG__Q?W&P#YH```\`"<1_D``#_6_\,&
MT<$``)`A0`"1X@,#D22@`)']>P&I_4,`D:,``/!C($?Y\U,"J?1#`I$H)$&I
M]5L#J?6C`I$FG$6I]V,$J?<S`I'Y:P6I^0,`JH$(0*U@`$#YX-<`^0``@-*`
M",`]````T"000/D`X!&1(S1`^>$#%ZI!"`"MA`H`^:,*`/E`"(`]Z"<)J>:?
M"JEE^O^7@```M`$`0#GS`P"J@2<`-:```/``A$?Y^W,&J>`[`/D``#_6H0``
M\"$81_D@`#_6P!$`M!P`0/F<$0"TN@``\%KO1_G@`QRJ0`,_U@#X?],`!`"1
MNP``\'NG1_E@`S_6^`,`JH0#0#GS`P"JQ"8`-+4``/"U\D?Y%@``T-;B%I'@
M`QBJXP,6J@(`@)(A`(!2H`(_UA@+`)&$'T`X)/__-0```-#V(P.1`(`5D0,`
M`!3`AD#XX`$`M.$#%ZHU^O^7^`,`JF#__[0!`$`Y(?__-'3Z_Y?@_O\TH0``
M\"&\1_G@`QBJ(``_UO8#`*J@`0"UN```\!B_1_F6`D#Y-@$`M,`"0#G@"@`T
MX`,6JF7Z_Y?`"0`UE@9`^90B`)$V__^UX`,6JD`#/];T`P"JX`,3JD`#/]:4
M`@`+E!(0$1P``-"X``#P&,]'^91^0)/@`Q2J8`,_UN0#%JKG`Q.JA8,2D08`
M`-`#``#0Q@`7D6,@%Y$"`("2]@,`JB$`@%*@`C_6X`,6J@$X@%(``S_6'P0`
M,>$``%2@``#P`(A'^0``/]8%`$"YOT0`<2$9`%2B``#P0K!'^>'#`Y'@`Q:J
M0``_UO,#`"J@```UX`-!N0`,%!(?$$!QP`@`5*```/``9$?Y`@``T",#0/GD
M`Q:J``!`^4)@&)&E``#PI91'^2$`@%*@`#_6%`"`TJ```/``($?YXM=`^0$`
M0/E"``'K`0"`TL$A`%3]>T&IX`,4JO-30JGU6T.I]V-$J?EK1:G[<T:I_\,&
MD<`#7];@`Q:JE"(`D0`#/];V`P"J`/7_M+'__Q<6`(#2K___%Z`"0/FS``#P
M<[Y'^2`!`+3A`Q>JQ?G_EV```+0!`$`Y`0(`-:`&0/FU(@"1(/__M;L``/![
MIT?YH`&`TAP``-"<PQ:1N@``\%KO1_E@`S_69`J`4O@#`*KS`P"J:___%V`"
M/];\`P"JX/W_M+H``/!:[T?Y0`,_U@#X?],`!`"16___%^`[0/GA"T&YX7L`
MN0``/];A>T"Y/P``:Z'V_U3@`T&Y`"``$A\`!W$A]O]4X`,4JF`#/]8!``#0
M]`,`JB#`&9'A`Q>JF/G_E^$#`*K`$`"T(`-`^2G\_Y?[`P"J0!``M.`#&ZI`
M`S_6X@,`JA\,`'%M`@!4HP``\&.T1_D`?$"3`!``T0$``-!@`P"+(>`9D>(/
M!ZE@`#_6(`$`->([0/E?$`!Q8!0`5$(4`%%@RV(X'[P`<>`3`%0?(`/5X0,7
MJA<``-#@`A21=_G_EZ```+0``$`Y'\``<0080'H!`0!4H@``\$+H1_G@`QNJ
M`0"`4D``/];[`P`JH`L`-;L``/![%T?YX`(4D0$``-`A8!:1[?K_EY>#$I$:
M``#0&0``T&`#/]8YHQJ1Y@,`*N4#%ZI'@QV1Y`,6JN`#%*H#``#0`@"`DF-@
M&I$A`(!2H`(_UKP``/"<BT?Y$@``%(`#/]8``$"Y'T0`<6$"`%1@`S_6<P8`
M$4&#'9'A`P#YY@,`*N<#$RKE`Q>JY`,6JN,#&:K@`Q2J`@"`DB$`@%*@`C_6
MX`,4J@$X@%(``S_6'P0`,6#]_U2A``#P(=A'^>`#%JH@`#_6X0,4J@```-``
MX!&1O?K_E^`#%*I3_O^74/__%Z```/``9$?YY`,6JB,#0/D"``#0``!`^4)@
M%Y&F``#PQI1'^2$`@%+``#_60___%T7^_Y>@``#P`"!'^>+70/D!`$#Y0@`!
MZP$`@-)A"@!4_7M!J>`#$ZKS4T*I]5M#J?=C1*GY:T6I_\,&D;```/`0OD?Y
M``(?UK4``/"U\D?YU?[_%SL#0/E^__\7\OG_E^,#`*I`]/^WI```\(0`1_D!
M*`#1N0``\#F31_D"`(!2X`,;*OK#!9'D#P>I@``_UN$#&JK@`QLJP@"`TB`#
M/];A<T&Y`&"(4N0/1ZD@:*AR/P``:\'Q_U3AZT)Y`*F(4C\``&M!\?]48<@`
MT?/C!9'@`QLJ`@"`4H``/];A`Q.JX`,;*@(%@-(@`S_6_X,&.>8#$ZJ%@Q*1
MY`,6JN`#%*H'``#0`P``T.>`'9%C(!J1`@"`DB$`@%*@`C_6HO__%SD'0/E9
M[/^TX`,9JD`#/]8?#`#QR>O_5.(_0/D`$`#1(`,`BP$``-`A`!J10``_UA\`
M`'%[$YF:5?__%Z```/``G$?Y```_UJ```/``G$?Y^W,&J0``/];]>[JI````
MT`"`$Y']`P"1\U,!J530.]6S``#P<VY'^?5;`JGV4P&1X0,6JO=C`ZGY(P#Y
MH@``\$(@1_F5:G/X0P!`^>,O`/D#`(#2HL)".:+&`CFR^/^7@```M`$`0#D_
MQ`!QP`P`5&W\_Y>5:G/XX`,5JK@``/`8)T?Y`0``T&,`@%(A`!N1(@"`4@`#
M/];U`P"JH`D`M*$*0/D"``#0@&IS^$(@&Y$A`$#YN0``\#DO1_D@`S_6H`I`
M^0$`0/D@#$"Y``NP-X!J<_@!``#08P"`4B%`&Y$B`(!2UP``D``#/];U`P"J
M8`$`M.$#%JH```#0`"`2D8GX_Y?B`P"J0`@`M*$*0/F`:G/X(0!`^2`#/]:C
M``#P8_A&^2(`@%*5:G/X`0``T"%@&Y'@`Q6J8``_U@(``-#A`P"J0L`;D>`#
M%:H@`S_6@FIS^/1"`I'A`Q:J````T`#@$9%?Q`(Y;_C_EX`*`/GS`P"J0`,`
MM*(``/!"S$?Y`3B`4N`#$ZI``#_6`00`$3\$`'$$B%%ZH04`5*```/``($?Y
MXB]`^0$`0/E"``'K`0"`TF$$`%3S4T&I]5M"J?=C0ZGY(T#Y_7O&J,`#7]:Y
M``#P.2]'^;[__Q>`!D#YR?W_EX`*`/GS`P"J8/S_M:$``/`AP$?Y````T`#@
M&Y$@`#_6``1`.8#S_S29__\7X$("D:(*0/D!!$#Y0`!`^2$`0/E\^?^7O?__
M%X!J<_BB``#P0JA'^4``/]:E__\7H```\`"<1_D``#_6]T("D>,#`"JD``#P
MA,!'^0```-#A!D#YX@,3J@"`')$A`$#Y@``_UOU[O:FA``#P(2!'^?T#`)'S
M4P&I\P,`JB(`0/GB%P#Y`@"`TG?[_Y?T`P"JH@``\$(X1_GA!8!20``_UA\`
M`/&A``#P(;Q'^8`&@)H@`#_6]`,`JN&3`)$```#0```4D13X_Y<@`0"T``!`
M.1_``'$$&$!ZX`>?&G\"`/'A!Y\:/P``:F$!`%2@``#P`"!'^>(70/D!`$#Y
M0@`!ZP$`@-(A`P!4\U-!J?U[PZC``U_68`)`.:#^_S2C``#P8UA'^0$``-#@
M`Q2J(0`7D8(`@-)@`#_6H/W_-:```/``($?YXA=`^0$`0/E"``'K`0"`TJ$`
M`%3@`Q.J\U-!J?U[PZA<^/\7H```\`"<1_D``#_6_X,*T0'@`)&C``#P8R!'
M^>)#`)']>P"I!.`!D?T#`)%1Y`!O\.4`;V4`0/GE3P'Y!0"`TH?D`&_C`P*J
M(P!`;2($P3QC.(`.001@;T`$6&]C""`N0@1(;R946$\`'#%.9*0@+R%42$_#
M'#!.`!RB3B$<)TX`'*-.!1RA3F2,GTR?``'KP?W_5/`?0:U"H`F1Y@]"K>'C
M`9'E$T.MX2/`/?'E`&\?(`/5(H#9/"%``)$@`-\\`!PB;@`<(VX`'#!N\!RG
M3L<<IDYF'*-.`E1!3P`$86^C'*5.A1RD3B0<H4Y`'*!.`1PQ3B&`@#P_``+K
MP?W_5`<H0*DD,X_2""1!J410J_+E/T&I(W2=TN+\6]/L_$+30A0'JHQY!ZHA
M`0C*(]NM\B$`"HHA``G*I0`$BR)`(HM!_4+3"Q!`^2%X"JHM?$"20@`%BZ4!
M",I"``L+YD-"J:4`!XJE``C*[P$$BT[\6].E``F+SA4"JB$`#,JE``^+(0`"
MBJ4`#@M/_$+3(0`-RN)Y`JK&``2+KOQ;TR$`"(O.%06J(0`&BX8!`LHA``X+
MC'U`DL8`!8JN_$+3Q@`,RA`"!(O&``V++_Q;T\UY!:KO%0&J[AM`^44`#<K&
M`!"+0GQ`DL8`#PNE``&*I0`"RC#\0M/.`02+I0`,B\_\6],,>@&J[Q4&JJ4`
M#HNE``\+H0$,RL_\0M,A``:*YGD&JJU]0)+NOT.I(0`-RB$``HNP_$+3P@$$
MBZ[\6],A``*+@@$&RD(`!8J,?4"2SA4%JD(`#,HN``X+!7H%JN$!!(M"``V+
M0@`!B\$`!<K0?$"2(0`.BB$`$,K-_5O3(0`,BZT5#JKOLT2I30`-"\;]0M/.
M>`ZJIGQ`DN(!!(NO_5O3(0`"BZ(`#LKO%0VJ0@`-BB\`#PNE_4+30@`&RH$!
M!(NE>`VJ0@`0BT(``8OL_5O3P0$%RO#]0M.,%0^J(0`/B@]Z#ZI,``P+[<-%
MJ<Y]0)(A``[*(0`&BX;]6]/&%`RJH@$$BQ`"!(LA``*+H@`/RBT`!@NE?$"2
M0@`,BH;]0M-"``7*QG@,JJ']6]-"``Z+0@`0BRP4#:JP_4+3X0$&RB$`#8H-
M>@VJ[L-&J4P`#`OO?4"2(0`/RB$`!8N%_5O3I10,JL(!!(L0`@2+(0`"B\(`
M#<HN``4+QGQ`DD(`#(J%_4+30@`&RJ5X#*K!_5O30@`/BT(`$(LL%`ZJT/U"
MTZ$!!<HA``Z*#GH.JN_#1ZE,``P+K7U`DB$`#<HA``:+AOU;T\84#*KB`02+
M$`($BR$``HNB``[*+P`&"Z5\0))"``R*AOU"TT(`!<K&>`RJX?U;TT(`#8M"
M`!"++!0/JO#]0M/!`0;*(0`/B@]Z#ZKMPTBI3``,"\Y]0)(A``[*(0`%BX7]
M6].E%`RJH@$$BQ`"!(LA``*+P@`/RBT`!0N%_4+3I7@,JD(`#(JA_5O3QGQ`
MDBP4#:KA`07*(0`-BN]]0))"``;*(0`/RD(`#HLA``:+[IM)J4(`$(M,``P+
ML/U"TPUZ#:J0_4+3P@$$BX[]6],A``*+SA4,JJ(`#<HN``X+I7Q`DD(`#(I"
M``7*#'H,JL$`!(M"``^+QOU;TT(``8OO5T#YH0$,RL84#JJM?4"2T@`""R$`
M#HHA``W*Y`$$BR$`!8O0_4+31?Y;TPYZ#JKO6T#Y(0`$BZ44$JI&0":+L``!
M"X0!#LJ$``9*X@$#BX0`#8L&_EO33?Y"TX0``HOB7T#YLGD2JL84$*HE0"6+
MSP`$"\$!$LHE``5*00`#BZ)`+(L,_D+3Y?U;TXUY$*I"``&+I10/JN%C0/FQ
M``(+AD`FBT0"#<J$``9*[/U"TR8``XN!0"Z+)/Y;TX]Y#ZHA``:+A!01JN9G
M0/F0``$+14`EBZ(!#\I%``5*+OY"T]%Y$:JL0#*+P@`#BP7^6]/F:T#Y@@$"
MBZ44$*HD0"2+X0$1RJX``@LD``1*#/Y"T\$``XN&0"V+D'D0JL$``8OF;T#Y
MQ/U;TT5`)8N$%`ZJ(@(0RHT``0M"``5*Q@`#BT)`+XO,_4+3CGD.JD(`!HOF
M<T#YI?U;TZ44#:HD0"2+`0(.RJ\``@LA``1*K/U"TXUY#:K&``.+(4`QB^3]
M6],A``:+14`EB^9W0/G"`0W*A!0/JD4`!4J1``$+K$`PB_#]0M/"``.+YGM`
M^0]Z#ZHE_EO3)$`DBZ$!#\J"`0*+I101JB0`!$JP``(+P0`#BRS^0M.&0"Z+
MC'D1JL$``8OF?T#Y14`EBP3^6]/B`0S*A!00JD(`!4J.``$+Q@`#BT)`+8L-
M_D+3\8-`^4(`!HNF>1"J)$`DB\7]6].!`0;*I10.JB$`!$JM``(+,`(#BR%`
M+XO/_4+3(0`0B^YY#JKPAT#YI/U;TX04#:I%0"6+L?U"T\(`#LJ/``$++7H-
MJD4`!4H"`@.+\(M`^21`)(O!`0W*K$`LBR0`!$KE_5O3I10/JH9`)HOQ_4+3
M`0(#B\$``8N"`0*+)GH/JJP``@M%0"6+H@$&RD(`!4J$_5O30D`NBX04#*KP
MNU&ID?U"TX\``0LL>@RJ)$`DB\$`#,KE_5O3(0`$2O']0M.E%`^J$`(#BR]Z
M#ZI"`!"+(4`MB^W'4JFP``(+S@$#BT5`)8L$_EO3(0`.BX04$*J"`0_*C@`!
M"T(`!4I"0":+I0$#BPW^0M-%``6+QOU;TZ)Y$*K&%`ZJ)$`DB^$!`LK-``4+
M)``$2B$"`XN$0"R+T?U"T_"?0/F$``&+K/U;TR%Z#JJ,%0VJID`FBT4``<J.
M`00+I0`&2A`"`XNE0"^+K_U"T^]Y#:JE`!"+\*-`^<;]6]/&%`ZJC$`LBR0`
M#\K-``4+C``,2M']0M,$`@.+@D$BB[#]6]-"``2+)'H.JJ9`)HOLIT#Y$!8-
MJN4!!,I"`!`+I0`&2J[]0M.A0"&+Q7D-JH8!`XM._$+3C``%JD/\6]-C%`*J
MC`$"BL)Y`JHA``:+[3M5J2$``PNC``0*AIN7THP!`ZIFX['R@T$OBR_\0M.L
M`0:++?Q;TZT5`:IC``R+K``"JF,`#0N,`0&*30`%"HP!#:KA>0&JS@$&BX1!
M)(MM_%O33``!JJT5`ZJ$``Z+A``-"XP!`XHM``(*;OQ"TXP!#:J-_%O3A4$E
MB\QY`ZKO.U:IHQ4$JBT`#*KO`0:+S@$&BZ4`#XN/_$+3HP`#"Z4!!(J-`0$*
MI0`-JFW\6].E0"*+XGD$JJT5`ZJ$`0*JI0`.BX0``XJE``T+30`,"H0`#:IM
M_$+3A$`ABZ%Y`ZKO-U>IKOQ;T\X5!:I#``&J8P`%BN\!!HNM`0:+A``/BZ_\
M0M.$``X++@`""F,`#JKE>06J8T`LBX[\6]/.%02J+``%JF,`#8N,`02*[<-`
M^6,`#@NN``$*C_Q"TXP!#JKD>02J@D$BBV[\6].L`0:+SA4#JNW'0/E"``R+
MK``$JD(`#@N,`0.*C@`%"HP!#JIO_$+3XWD#JJT!!HN!02&+3OQ;T\X5`JJ,
M``.J(0`-BXP!`HHA``X+;@`$"HP!#JI._$+3PGD"JH5!)8OM.UFI+_Q"TZP!
M!HLM_%O3K14!JJ4`#(ML``*JI0`-"XP!`8I-``,*C`$-JN%Y`:K.`0:+A$$D
MBZW\6]-,``&JK14%JH0`#HN$``T+C`$%BBT``@JN_$+3C`$-JHW\6].#02.+
MS'D%JN\[6JFM%02J)0`,JJ4`!(KO`0:+S@$&BV,`#XN/_$+38P`-"XT!`0JE
M``VJY'D$JJ)`(HMM_%O3K14#JH4!!*I"``Z+I0`#BD(`#0N-``P*I0`-JFW\
M0M.A0"&+I7D#JN\W6ZE._%O3SA4"JH,`!:IC``**[P$&BZT!!HLA``^+KP`$
M"B$`#@MC``^J3OQ"TVQ`+(O">0*J(_Q;TXP!#8MC%`&JK0`"JB_\0M.#`0,+
MK`$!BN%Y`:I-``4*[C]<J8P!#:J$022+;?Q;TZT5`ZK,`0:++@`""H0`#(M,
M``&JA``-"XP!`XJ,`0ZJ;?Q"TZUY`ZKO`0:+A4$EBX[\6]/LZT#YHP`/B\X5
M!*HE``VJ8P`."Z4`!(JN`0$*C_Q"TZ4`#JIN_%O3HD`BBX4!!HOD>02JSA4#
MJD(`!8NE`02J0@`."V[\0M.E``.*PWD#JNR[7:F/``T*I0`/JD_\0M.E0"&+
M0?Q;TR$4`JJ,`0:+I0`,BXP``ZJE``$+C`$"BF$`!`KB>0*JC`$!JE```PJ!
M02V+S0$&BZS\6],A``V+[_=`^6T``JJ,%06JK0$%BHX!`0NM`1"JI$$DB[#\
M0M/M`0:+QOU;T^_[0/F$``V+QA0.J@UZ!:K1``0++$`LBT$`#<K%.IC2(0`,
M2D5,N?(A0".+[`$%B]#]0M,C_EO3[_]`^0YZ#JIC%!&J(0`,BVP``0N&0":+
MI`$.RB-`(XN$``9*Y@$%BX)`(HLO_D+3A/U;T_%Y$:I"``:+A!0,JN8#0?F2
M``(+P0$1RD1`)(LC``-*P0`%BV9`+8N-_4+30_Y;T[!Y#*K&``&+8Q02JN$'
M0?EO``8+(@(0RDS^0M-"``1*)``%BT)`+HOA_5O3C7D2JD0`!(OB"T'Y(10/
MJBX`!`O#0".+!@(-RNS]0M/#``-*1@`%BV-`,8O"_5O3CWD/JF,`!HOF#T'Y
M0A0.JE$``PN!0"&+I`$/RF)`(HN!``%*Q``%BRQ`,(O0_4+3(?Y;TPYZ#JKF
M$T'YA`$$BR$4$:KC`0[*,``$"V(``DHL_D+3PP`%BT9`+8L"_EO3C'D1JL,`
M`XOF%T'Y0A00JDT``PN!0"&+Q`$,RA'^0M.!``%*Q@`%BR1`+XNA_5O3[QM!
M^80`!HLA%`VJ)GH0JF)`(HLP``0+@P$&RN\!!8MB``)*L?U"TT-`+HL"_EO3
M[A]!^2UZ#:IC``^+0A00JD\``PN!0"&+Q``-RF)`(HN!``%*Q`$%BRQ`+(L.
M_D+3X?U;T]!Y$*J$`02+(10/JNPC0?DQ``0+HP$0RH%`(8MB``)*@P$%BT9`
M)HOL_4+3(OY;TX]Y#ZK#``.+0A01JN8G0?E.``,+!`(/RF)`(HN!``%*Q@`%
MBR1`+8LM_D+3P?U;TX0`!HOL*T'YIGD1JB$4#JKC`0;*+0`$"V(``DJ,`06+
M0T`PB]#]0M.B_5O3#GH.JF,`#(OL+T'Y0A0-JE$``PN!0"&+Q``.RF)`(HN!
M``%*A`$%BRQ`+XNO_4+3(?Y;T^UY#:J$`02+(101JNPS0?DP``0+PP$-RB_^
M0M-B``)*C`$%BT)`)HL#_EO31@`,BV,4$*KB-T'Y['D1JH%`(8MO``8+I`$,
MRL-`(XN$``%*00`%BX1`+HOB_5O3#OY"TX0``8OQ.T'YP7D0JD(4#ZJ&`0'*
M3@`$"\,``THP`@6+9D`MB_']0M/#_5O3[3]!^6,4#JHO>@^JQ@`0BW``!@N"
M0"*+)``/RL9`(XN$``)*H@$%BXQ`+(O-_4+3!/Y;TXP!`HOB0T'YA!00JJYY
M#JJ#``P+[0$.RN=`+XNM`09*1@`%BZ%!(8MB_%O30A0#JB$`!HOM1T'Y(0`"
M"P+^0M.&022+0G@0JB3\6]/,`0+*A!0!JHP!!DJF`06+9?Q"T\8`!XNC>`.J
MA``,BZ<``+#G($?Y(0`*"T(`"0O%`0L+A``&"V,`"`L$!`"I`P@!J040`/GB
M3T'YX0!`^4(``>L!`(#2@0``5/U[0*G_@PJ1P`-?UJ```+``G$?Y```_UJ$`
M`+`AI$?Y_7N_J0`0@-+]`P"1(``_UA\8`/DB``"0(0``D!]X`+E!5,$](@``
MD")0P3U`6,$]`@0`K0`(@#W]>\&HP`-?UA\@`]4?(`/5_7N[J4)\0)/]`P"1
M\U,!J13@`)'S`P&J]5L"J?4#`*KV`P*J]V,#J?DC`/D%C$*I`'A`N:0,`@N_
M``3K8Y2#FF)T0HNDB@*IX`,`-;@``+`8*T?YW_X`<>T%`%36`@%1V7X&4S<'
M`!'W:GK3=P(7BV`&0*W@`Q6J@`8`K6`&0:US`@&1@`8!K9K[_Y=_`A?K`?__
M5-8:&4OA`Q>JX`,4JL)^0),``S_6MGH`N?-30:GU6T*I]V-#J?DC0/G]>\6H
MP`-?UA<(@%+W`@!+_P(6:X#"((NX``"P&"M'^??2EAKY?D"3X@,9J@`#/]:@
M>D"YX`(`"Z!Z`+D?``%Q@?W_5-8"%TMS`AF+X`,5JGG[_Y?1__\7]P,3JN#_
M_Q<?(`/5_7N\J>(/@!+]`P"1\U,!J?0#`*KS`P&J]5L"J3;@`)'H&P#]-11`
M^2@80/V@(@-3`00`$<+*(#C`PB&+/^``<<T+`%2C``"P8_Q'^0((@%)"``%+
M`0"`4D)\0)-@`#_6X`,3JEO[_Y=_-@#Y``0`3\`"`*W`"H`]!05P?P`%:'\$
M!7A_H0)GGN`#$ZHC!&A_(@1P?Z`$`VXA!'A_@`0%;@`%!VY@!`EN0`0+;B`$
M#6Z@'@].8#H`_4;[_Y=A`D#YX`,3JK```+`0VD?Y(?Q8TX$"`#EA`D#Y(?Q0
MTX$&`#EA`D#Y(?Q(TX$*`#EA`D#Y@0X`.6$&0/DA_%C3@1(`.6$&0/DA_%#3
M@18`.6$&0/DA_$C3@1H`.6$&0/F!'@`Y80I`^2'\6-.!(@`Y80I`^2'\4-.!
M)@`Y80I`^2'\2-.!*@`Y80I`^8$N`#EA#D#Y(?Q8TX$R`#EA#D#Y(?Q0TX$V
M`#EA#D#Y(?Q(TX$Z`#EA#D#Y@3X`.6$20/DA_%C3@4(`.6$20/DA_%#3@48`
M.6$20/DA_$C3@4H`.6$20/F!3@`Y\U-!J?5;0JGH&T#]_7O$J``"']:C``"P
M8_Q'^0('@%)"``%+`0"`4D)\0)-@`#_6J?__%Q\@`]4?(`/5HP``T+```+`0
MCD?Y8P1`^0`"']8?(`/5_7N_J?T#`)']>\&HP`-?U@```````````0`"````
M```E<R\E<P```"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y
M;F%,;V%D97(``````$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````&5X90``````
M+W!R;V,O)6DO)7,``````"X`````````4$%27U1%35```````````%!!4E]0
M4D]'3D%-10`````Z`````````"\`````````)7,E<R5S``!015),-4Q)0@``
M````````4$523$Q)0@!015),-4]05```````````4$523$E/``!005)?24Y)
M5$E!3$E:140`4$%27U-005=.140``````%!!4E]#3$5!3@````````!005)?
M1$5"54<`````````4$%27T-!0TA%`````````%!!4E]'3$]"04Q?1$5"54<`
M`````````%!!4E]'3$]"04Q?5$U01$E2`````````%!!4E]435!$25(`````
M``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``````````,0``
M``````!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````'!A<BT`````
M)7,E<R5S)7,``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET
M(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*````````4$%42`````!P87)L
M`````"YP87(`````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E
M<R5S=&5M<"TE=2TE=25S```````````8`````````'!E<FP`````,```````
M``!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`U.````%5N86)L92!T;R!C
M<F5A=&4@8V%C:&4@9&ER96-T;W)Y```````````E<SH@8W)E871I;VX@;V8@
M<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE
M:2D*```````````M90```````"TM````````55-%4@````!54T523D%-10``
M````````5$U01$E2``!414U01$E2`%1%35``````5$U0```````O=&UP````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D
M8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2
M+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*
M)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S
M7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*
M;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE
M(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@
M)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@
M;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@
M)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H
M*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?
M34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D
M8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F
M:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?
M96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?
M<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S
M96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@
M>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO
M=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C
M(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA
M<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?
M1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD
M(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE
M(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P
M;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A
M9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B
M7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C
M:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H
M92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,
M15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I
M.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?
M1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B
M"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA
M;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B
M+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX
M?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@
M*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@
M*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI
M)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M
M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L
M;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@
M7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF
M+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M
M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U
M<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII
M9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W
M*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#
M86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD
M9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T
M=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@
M/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N
M("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD
M(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E
M<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA
M;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA
M;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H
M)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII
M9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA
M;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@
M*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@
M)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P
M87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N
M7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=
M(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%
M55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E
M:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=6
M6S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF
M("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@
M*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G
M02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G
M3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L
M92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G
M86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS
M:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE
M('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D
M,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD[
M"FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG
M4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[
M(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I
M<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`
M24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@
M*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O
M<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE
M<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII
M9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY
M;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I]
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I
M<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#
M.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/
M.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ
M1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H
M+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H
M9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D
M;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB
M*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T
M>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@
M/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A
M8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X]
M(")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@
M)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L
M;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B
M.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@
M>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO
M=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@
M*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T
M;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X
M="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L
M<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H
M)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@
M<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE
M(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER
M.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM
M>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS
M965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H
M+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C
M86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@
M*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[
M4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C
M86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-
M4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H
M<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R
M;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA
M;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?
M<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!
M4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S
M+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?
M<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!
M345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@
M>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@
M)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I
M"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H
M*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<
M+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D
M14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"
M04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[
M4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*
M:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP
M<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I
M9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF
M:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP
M('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP
M86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2
M.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A
M<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@
M<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE
M.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/
M4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@
M)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS:2YC``````````!#04-(
M10```2-%9P````")J\WO`````/[<NI@`````=E0R$`````#PX=+#````````
M`````````1L#.P0!```?````4(;__^0&``#0B/__'`$``"")__\P`0``4(G_
M_T0!``"0B?__6`$``.")__]\`0``\(G__Y`!````B___!`(``+"+__\L`@``
M@(W__Y`"``#@CO__%`,``!"0__],`P``<)+__YP#``#PDO__P`,``"23___4
M`P``0)3__P@$``!$E/__'`0```"7__^@!```0)?__\0$``#PF/__-`4``,"=
M__]P!0``0)[__Y`%``!PGO__L`4``-"?___T!0``U*;__W0&``!`J?__L`8`
M`%"J__\8!P``$+O__SP'``!@N___7`<``("\__^4!P``8+[__]`'````````
M$``````````!>E(`!'@>`1L,'P`0````&````*R'__\T`````$$''A`````L
M````Z(?__S``````````$````$`````$B/__/``````````@````5````#"(
M__](`````$$.()T$G@-"DP).WMW3#@`````0````>````%R(__\$````````
M`'````",````6(C__P@!````00Y0G0J>"4*3")0'0I@#EP1$F0)#E@65!E?6
MU4'8UT'90][=T]0.`$$.4),(E`>5!I8%EP28`YD"G0J>"4_6U4'8UT'90=[=
MT]0.`$$.4),(E`>="IX)0I<$F`.9`D'8UT+9)``````!``#TB/__K`````!!
M#K`!G1:>%423%%@*WMW3#@!!"P```&`````H`0``?(G__]`!````0P[0`9T:
MGAE&E!>3&$34TTC>W0X`00[0`9,8E!>=&IX91I85E190F!.7%&O8UTP*U--!
MUM5!"T:7%)@30]/4U=;7V$.4%Y,80985E19!F!.7%`"`````C`$``.B*__]8
M`0```$$.@(`$0@[@@`1%G8Q`GHM`0Y.*0)2)0$>9A$!)EH=`E8A`19B%0)>&
M0%G6U4+8UTS=WMG3U$$.@(`$00X`00[@@`23BD"4B4"5B$"6AT"7AD"8A4"9
MA$"=C$">BT!$UM5!V-=%EH=`E8A`09B%0)>&0``T````$`(``,2+__\H`0``
M`$(.P"!$G8@$GH<$0I.&!)2%!$.5A`26@P1F"MW>U=;3U`X`00L``$P```!(
M`@``O(S__UP"````00YPG0Z>#4*3#)0+294*E@F7")@'F0::!0)("M[=V=K7
MV-76T]0.`$$+29L$80K;00M,VT*;!$/;0YL$0ML`(````)@"``#,CO__>```
M``!!#B"=!)X#1),"E`%0WMW3U`X`$````+P"```HC___-``````````P````
MT`(``$B/__\4`0```$(.T$!$G8H(GHD(0I.(")2'"$*5A@AU"MW>U=/4#@!!
M"P``$`````0#```PD/__!`````````"`````&`,``""0__^X`@```$(.D")#
MG:($GJ$$0I.@!)2?!$*5G@26G01@"MW>U=;3U`X`00M#F)L$EYP$1)R7!)N8
M!$J:F029F@0"0PK8UT':V4'<VT$+6PK8UT':V4'<VT$+1MG:0=C70=S;1)B;
M!)><!$&:F029F@1!G)<$FY@$```@````G`,``%B2__\\`````$$.()T$G@-%
MDP)&WMW3#@````!L````P`,``'22__^H`0```$(.P"!#G8@$GH<$1I6$!$:4
MA023A@1LU--!U4'=W@X`00[`()6$!)V(!)Z'!$'53MW>#@!!#L`@DX8$E(4$
ME80$G8@$GH<$3`K4TT+51`M#T]350Y2%!).&!$&5A`0`.````#`$``"TD___
MS`0```!!#F"=#)X+1),*E`E&E0B6!Y<&F`69!`,;`0K>W=G7V-76T]0.`$$+
M````'````&P$``!(F/__>`````!##B"=!)X#50K>W0X`00L<````C`0``*B8
M__\P`````$,.$)T"G@%(WMT.`````$````"L!```N)C__V`!````00Y0G0J>
M"423")0'0I4&E@58"M[=U=;3U`X`00M/"M[=U=;3U`X`00M!EP1@UT27!```
M?````/`$``#4F?__!`<```!!#K`#19TTGC-$DS*4,4.5,)8O0Y<NF"U"F2R:
M*UF<*9LJ`HD*W-M!W=[9VM?8U=;3U`X`00L"K=O<3]W>V=K7V-76T]0.`$,.
ML`.3,I0QE3"6+Y<NF"V9+)HKFRJ<*9TTGC,"2-O<0YPIFRH````X````<`4`
M`%B@__]L`@```$$.8)T,G@M$DPJ4"425")8'1)<&F`69!`)D"M[=V=?8U=;3
MU`X`00L````P````K`4``(BB__\0`0```$$.,)T&G@5$DP24`V8*WMW3U`X`
M00M4"M[=T]0.`$$+````,````.`%``!D?___;`(```!!#F"=#)X+19,*E`F5
M")8'`FD*WMW5UM/4#@!!"P```````"`````4!@``,*/__\`0````00Z@!4:=
M5)Y3`R4$"MW>#@!!"QP````X!@``S+/__T@`````0PX0G0*>`4[>W0X`````
M-````%@&``#\L___'`$```!!#E"="IX)0Y,(E`=#E0:6!427!)@#F0)E"M[=
MV=?8U=;3U`X`00LX````D`8``.2T___8`0```$$.0)T(G@=#DP:4!4.5!)8#
M0@5(`@)D"M[=U=;3U`9(#@!!"P`````````0````S`8``(BV__\4````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````4!\`````````'P````````$`````````
M[`(````````!`````````/<"`````````0`````````!`P````````\`````
M````@@0````````,`````````$`;````````#0````````#D4P```````!D`
M````````>*L!```````;``````````@`````````&@````````"`JP$`````
M`!P`````````"`````````#U_O]O`````)@"````````!0````````"(#0``
M``````8`````````^`,````````*`````````*4$````````"P`````````8
M`````````!4````````````````````#`````````*BM`0```````@``````
M``!(`````````!0`````````!P`````````7`````````/@:````````!P``
M``````!@$P````````@`````````F`<````````)`````````!@`````````
M&````````````````````/O__V\``````0``"`````#^__]O```````3````
M````____;P`````"`````````/#__V\`````+A(```````#Y__]O``````\`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````!@&P```````&`;````````8!L```````"(
MJP$`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````#04P``````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````,`;````
M``````````````````````````BP`0``````"E!!4BYP;0H``````````+!7
M````````<%0``````````````````&!5````````D%<```````"85P``````
M`*!7````````J%<``````````````````'A7````````@%<`````````````
M`````+A7````````1T-#.B`H1TY5*2`Q,BXQ+C``1T-#.B`H8W)O<W-T;V]L
M+4Y'(#$N,C4N,"XR.%]B.#AD,S,X*2`Q,BXQ+C```"YS:'-T<G1A8@`N:6YT
M97)P`"YN;W1E+F=N=2YB=6EL9"UI9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S
M:``N9'EN<WEM`"YD>6YS='(`+F=N=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R
M`"YR96QA+F1Y;@`N<F5L82YP;'0`+FEN:70`+G1E>'0`+F9I;FD`+G)O9&%T
M80`N96A?9G)A;65?:&1R`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A
M<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`````````
M````````````````````````````````````````````````````````````
M```````````````````````````+`````0````(`````````.`(````````X
M`@```````!L````````````````````!````````````````````$P````<`
M```"`````````%0"````````5`(````````D````````````````````!```
M`````````````````"8````'`````@````````!X`@```````'@"````````
M(`````````````````````0````````````````````T````]O__;P(`````
M````F`(```````"8`@```````%P!````````!0`````````(````````````
M````````/@````L````"`````````/@#````````^`,```````"0"0``````
M``8````#````"``````````8`````````$8````#`````@````````"(#0``
M`````(@-````````I00```````````````````$```````````````````!.
M````____;P(`````````+A(````````N$@```````,P`````````!0``````
M```"``````````(`````````6P```/[__V\"```````````3`````````!,`
M``````!@``````````8````"````"````````````````````&H````$````
M`@````````!@$P```````&`3````````F`<````````%``````````@`````
M````&`````````!T````!````$(`````````^!H```````#X&@```````$@`
M````````!0```!4````(`````````!@`````````?@````$````&````````
M`$`;````````0!L````````8````````````````````!```````````````
M`````'D````!````!@````````!@&P```````&`;````````4```````````
M`````````!````````````````````"$`````0````8`````````P!L`````
M``#`&P```````"0X``````````````````!`````````````````````B@``
M``$````&`````````.13````````Y%,````````4````````````````````
M!````````````````````)`````!`````@``````````5`````````!4````
M````<$$``````````````````!````````````````````"8`````0````(`
M````````<)4```````!PE0````````0!```````````````````$````````
M````````````I@````$````"`````````'B6````````>)8```````#@!@``
M````````````````"````````````````````+`````.`````P````````!X
MJP$``````'BK````````"`````````````````````@`````````"```````
M``"\````#P````,`````````@*L!``````"`JP````````@`````````````
M```````(``````````@`````````R`````8````#`````````(BK`0``````
MB*L````````@`@````````8`````````"``````````0`````````-$````!
M`````P````````"HK0$``````*BM````````6`(```````````````````@`
M````````"`````````#6`````0````,``````````+`!````````L```````
M`(@````````````````````(````````````````````W`````@````#````
M`````(BP`0``````B+`````````H````````````````````"```````````
M`````````.$````!````,````````````````````(BP````````/P``````
M``````````````$``````````0`````````!`````P``````````````````
M``````#'L````````.H````````````````````!````````````````````
